<?php

namespace App\Services;

use Illuminate\Http\UploadedFile;
use Google\Service\Drive\DriveFile;
use Google\Service\Drive\Permission;
use Google\Service\Drive as GoogleServiceDrive;

class GoogleDriveService
{
    protected $drive;

    public function __construct(GoogleServiceDrive $drive)
    {
        $this->drive = $drive;
    }

    /**
     * Create a folder in Google Drive
     */
    public function createFolder(string $folderName, ?string $parentFolderId = 'root'): string
    {
        $fileMetadata = new DriveFile([
            'name' => $folderName,
            'mimeType' => 'application/vnd.google-apps.folder',
            'parents' => [$parentFolderId],
        ]);

        $folder = $this->drive->files->create($fileMetadata, [
            'fields' => 'id',
        ]);

        return $folder->id;
    }

    /**
     * Upload a file to Google Drive
     */
    public function uploadFile(UploadedFile $file, ?string $folderId = 'root'): array
    {

        $fileMetadata = new DriveFile([
            'name' => $file->getClientOriginalName(),
        ]);

        if ($folderId) {
            $fileMetadata->setParents([$folderId]);
        }

        $content = file_get_contents($file->getRealPath());
        $uploadedFile = $this->drive->files->create($fileMetadata, [
            'data' => $content,
            'mimeType' => $file->getMimeType(),
            'uploadType' => 'multipart',
            'fields' => 'id, name, mimeType, modifiedTime, size, webViewLink, parents',
        ]);

        // Make the file publicly accessible for viewing
        $this->setFilePermission($uploadedFile->id);

        return [
            'id' => $uploadedFile->id,
            'name' => $uploadedFile->name,
            'mimeType' => $uploadedFile->mimeType,
            'modifiedTime' => $uploadedFile->modifiedTime,
            'size' => $uploadedFile->size,
            'webViewLink' => $uploadedFile->webViewLink,
            'parents' => $file->parents ?? [],
        ];
    }

    /**
     * Set file permission to be publicly viewable
     */
    protected function setFilePermission(string $fileId): void
    {
        $permission = new Permission([
            'type' => 'anyone',
            'role' => 'reader',
        ]);

        $this->drive->permissions->create($fileId, $permission);
    }

    /**
     * List files in a folder
     */
    public function listFiles(?string $folderId = 'root', $pageSize = 100): array
    {
        $query = "'{$folderId}' in parents";

        $response = $this->drive->files->listFiles([
            'q' => $query,
            'pageSize' => $pageSize,
            'fields' => 'files(id, name, modifiedTime, size, mimeType, webViewLink, webContentLink, parents)',
            'orderBy' => 'folder,name'
        ]);

        return array_map(function ($file) {
            return [
                'id' => $file->id,
                'name' => $file->name,
                'modifiedTime' => $file->modifiedTime,
                'size' => $file->size,
                'mimeType' => $file->mimeType,
                'webViewLink' => $file->webViewLink,
                'webContentLink' => $file->webContentLink,
                'parents' => $file->parents ?? [],
            ];
        }, $response->getFiles());
    }

     /**
     * Rename file or folder
     */
    public function renameFile(string $fileId,string $newName): array
    {
        $fileMetadata = new DriveFile([
            'name' => $newName
        ]);

        $updatedFile = $this->drive->files->update($fileId, $fileMetadata, [
            'fields' => 'id, name, mimeType, modifiedTime, size, webViewLink, parents'
        ]);

        return [
            'id' => $updatedFile->id,
            'name' => $updatedFile->name,
            'mimeType' => $updatedFile->mimeType,
            'modifiedTime' => $updatedFile->modifiedTime,
            'size' => $updatedFile->size,
            'webViewLink' => $updatedFile->webViewLink,
            'parents' => $file->parents ?? [],
        ];
    }

    /**
     * Move a file to a different folder
     */
    public function moveFile(string $fileId, ?string $newParentId = 'root'): array
    {
        $file = $this->drive->files->get($fileId, ['fields' => 'parents']);
        $previousParents = join(',', $file->parents);

        $updatedFile = $this->drive->files->update($fileId, new DriveFile(), [
            'addParents' => $newParentId,
            'removeParents' => $previousParents,
            'fields' => 'id, name, mimeType, modifiedTime, size, webViewLink, parents',
        ]);

        return [
            'id' => $updatedFile->id,
            'name' => $updatedFile->name,
            'mimeType' => $updatedFile->mimeType,
            'modifiedTime' => $updatedFile->modifiedTime,
            'size' => $updatedFile->size,
            'webViewLink' => $updatedFile->webViewLink,
            'parents' => $file->parents ?? [],
        ];
    }

    /**
     * Delete a file or folder
     */
    public function delete(string $fileId): bool
    {
        try {
            $this->drive->files->delete($fileId);
            return true;
        } catch (\Exception $e) {
            return false;
        }
    }
    
    /**
     * Get file information
     */
    public function getFile(string $fileId): ?array
    {
        try {
            $file = $this->drive->files->get($fileId, [
                'fields' => 'id, name, modifiedTime, size, mimeType, webViewLink, parents'
            ]);
            
            return [
                'id' => $file->id,
                'name' => $file->name,
                'mimeType' => $file->mimeType,
                'modifiedTime' => $file->modifiedTime,
                'size' => $file->size,
                'webViewLink' => $file->webViewLink,
                'parents' => $file->parents ?? [],
            ];
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * Get all folders in the drive
     */
    public function getAllFolders(?string $parentFolderId = 'root'): array
    {
        $folders = [];
        $query = "mimeType = 'application/vnd.google-apps.folder' and '{$parentFolderId}' in parents";
    
        $response = $this->drive->files->listFiles([
            'q' => $query,
            'fields' => 'files(id, name, parents)',
            'orderBy' => 'name',
        ]);

        foreach ($response->getFiles() as $folder) {
            $folders[] = [
                'id' => $folder->id,
                'name' => $folder->name,
                'subfolders' => $this->getAllFolders($folder->id),
            ];
        }

    
        return $folders;
    }
}