<?php

namespace App\Notifications;

use App\Models\User;
use Illuminate\Support\Str;
use Illuminate\Bus\Queueable;
use Illuminate\Notifications\Notification;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Contracts\Queue\ShouldQueue;

class UserNotification extends Notification implements ShouldQueue
{
    use Queueable;

    protected $userId;
    protected $userName;
    protected $actionType;
    protected $reason;
    protected $role;

    /**
     * Create a new notification instance.
     */
    /**
     * Create a new notification instance.
     *
     * @param int $userId
     * @param string $userName
     * @param string $actionType
     * @param string $role
     * @param string|null $reason
     */
    public function __construct($userId, $userName, string $actionType, string $role, ?string $reason = null)
    {
        $this->userId = $userId;
        $this->userName = $userName;
        $this->actionType = $actionType;
        $this->role = $role;
        $this->reason = $reason;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function via($notifiable)
    {
        return ['mail'];
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return \Illuminate\Notifications\Messages\MailMessage
     */
    public function toMail($notifiable)
    {

        return (new MailMessage)
            ->subject($this->getSubject())
            ->line($this->getContent())
            ->when($this->reason, function ($mail) {
                return $mail->line("Raison de la désactivation : {$this->reason}");
            })
            ->action('Voir le profil de l’utilisateur', url("{$this->getActionUrl()}"));
    }

    private function getActionUrl()
    {
        $baseUrl = env('FRONTEND_URL');
        return "{$baseUrl}/admin/{$this->role}s/{$this->userId}";
    }

    private function getContent()
    {
        $contents = [
            'new_account' => "Un nouveau compte étudiant a été créé",
            'deactivated_account' => "L'utilisateur " . Str::title($this->userName) . " a désactivé son compte.",
            'reactivated_account' => "L'utilisateur " . Str::title($this->userName) . " a réactivé son compte.",
        ];

        return $contents[$this->actionType] ? $contents[$this->actionType] : 'Détails de la notification indisponibles.';
    }

    private function getSubject()
    {
        $subjects = [
            'new_account' => 'Création d’un nouveau compte étudiant',
            'deactivated_account' => 'Désactivation d\'un compte utilisateur',
            'reactivated_account' => 'Réactivation d\'un compte utilisateur',
        ];

        return $subjects[$this->actionType] ?? 'Alerte de notification';
    }
}
