<?php

namespace App\Notifications;

use Carbon\Carbon;
use App\Models\User;
use App\Models\Subscription;
use Illuminate\Bus\Queueable;
use Illuminate\Broadcasting\Channel;
use Illuminate\Notifications\Notification;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Contracts\Broadcasting\ShouldBroadcast;

class subscriptionNotification extends Notification implements ShouldQueue, ShouldBroadcast
{
    use Queueable;

    public $subscription;
    public $action;
    public $recieverId;

    /**
     * Create a new notification instance.
     */
    public function __construct(Subscription $subscription, string $action, $recieverId)
    {
        $this->subscription = $subscription;
        $this->action = $action;
        $this->recieverId = $recieverId;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array
     */
    public function via($notifiable)
    {
        $channels = [];
        $channels[] = 'mail';

        if ($notifiable->notificationPref->in_app_notifications) {
            $channels[] = 'database';
            $channels[] = 'broadcast';
        }

        return $channels;
    }

    /**
     * Get the mail representation of the notification.
     *
     * @return MailMessage
     */
    public function toMail($notifiable)
    {
        $message = (new MailMessage())->subject($this->getSubject());
        $daysRemaining = now()->diffInDays($this->subscription->end_date);
        $reciver = User::find($this->recieverId);
        $subscriptionEndDate = Carbon::parse($this->subscription->end_date)
                            ->timezone($reciver->timezone)
                            ->format('d-m-Y, H:i');
                            
        $baseUrl = env('FRONTEND_URL');

        switch ($this->action) {
            case 'activated':
                $message->line('Votre abonnement a été activé avec succès!')
                        ->line('Plan : ' . $this->subscription->subscriptionPlan->name)
                        ->line('Crédits : ' . $this->subscription->credits_remaining)
                        ->line('Date de fin : ' . $subscriptionEndDate)
                        ->action('Voir mon abonnement', url("{$baseUrl}/student/subscription"));
                break;
            case 'subscription_with_lesson':
                $message->line('Un étudiant vient de souscrire à son premier abonnement et nécessite une leçon d\'essai.')
                    ->line('Étudiant : ' . $this->subscription->student->first_name . ' ' . $this->subscription->student->last_name)
                    ->line('Plan souscrit : ' . $this->subscription->subscriptionPlan->name)
                    ->line('Date de souscription : ' . $this->subscription->created_at->format('d-m-Y'))
                    ->action('Programmer la leçon d\'essai', url("{$baseUrl}/admin/lessons"));
                break;
            case 'subscription_expiring_soon':
                $message->line("Votre abonnement arrive bientôt à expiration. Il vous reste {$daysRemaining} jour(s). Pensez à le renouveler.")
                    ->line('Plan : ' . $this->subscription->subscriptionPlan->name)
                    ->line('Crédits restants : ' . $this->subscription->credits_remaining)
                    ->line('Date de fin : ' . $subscriptionEndDate)
                    ->action('Renouveler mon abonnement', url("{$baseUrl}/student/subscription"));
                break;
            case 'subscription_expired':
                $message->line('Votre abonnement a expiré.')
                    ->line('Plan : ' . $this->subscription->subscriptionPlan->name)
                    ->line('Crédits restants : 0')
                    ->action('Renouveler mon abonnement', url("{$baseUrl}/student/subscription"));
                break;
            case 'subscription_auto_renewed':
                $message->line('Votre abonnement a été renouvelé automatiquement.')
                    ->line('Plan : ' . $this->subscription->subscriptionPlan->name)
                    ->line('Nouveaux crédits : ' . $this->subscription->credits_remaining)
                    ->line('Nouvelle date de fin : ' . $subscriptionEndDate)
                    ->action('Voir mon abonnement', url("{$baseUrl}/student/subscription"));
                break;
            case 'subscription_auto_renewal_failed':
                $message->line('Le renouvellement automatique de votre abonnement a échoué.')
                    ->line('Plan : ' . $this->subscription->subscriptionPlan->name)
                    ->line('Veuillez mettre à jour votre méthode de paiement par défaut')
                    ->action('Mettre à jour le paiement', url("{$baseUrl}/student/subscription"));
                break;
            case 'subscription_payment_failed':
                $message->line('Le paiement de votre abonnement a échoué.')
                    ->line('Plan : ' . $this->subscription->subscriptionPlan->name)
                    ->line('L\'abonnement a été annulé.')
                    ->line('Veuillez mettre à jour votre méthode de paiement et réessayer.')
                    ->action('Réessayer', url("{$baseUrl}/student/subscription"));
                break;
            case 'canceled':
                $message->line('Votre abonnement a été manuellement annulé par un administrateur.')
                    ->line('Plan : ' . $this->subscription->subscriptionPlan->name)
                    ->line('Date d\'annulation : ' . now()->format('d-m-Y, H:i'))
                    ->line('Crédits restants : 0')
                    ->action('Voir mon abonnement', url("{$baseUrl}/student/subscription"))
                    ->line('Si vous pensez qu\'il s\'agit d\'une erreur, veuillez contacter le support à l\'adresse : ' . env('COMPANY_EMAIL'));
                break;
            case 'suspended':
                $message->line('Votre abonnement a été suspendu par un administrateur.')
                    ->line('Plan : ' . $this->subscription->subscriptionPlan->name)
                    ->line('Date de suspension : ' . now()->format('d-m-Y, H:i'))
                    ->action('Voir mon abonnement', url("{$baseUrl}/student/subscription"))
                    ->line('Si vous pensez qu\'il s\'agit d\'une erreur, veuillez contacter le support à l\'adresse : ' . env('COMPANY_EMAIL'));
                break;
            case 'reactivation_after_suspension':
                $message->line('Votre abonnement a été réactivé après suspension.')
                    ->line('Plan : ' . $this->subscription->subscriptionPlan->name)
                    ->line('Crédits restants : ' . $this->subscription->credits_remaining)
                    ->line('Date de fin : ' . $subscriptionEndDate)
                    ->action('Voir mon abonnement', url("{$baseUrl}/student/subscription"));
                break;
            case 'updated':
                $message->line('Votre abonnement a été mis à jour.')
                    ->line('Plan : ' . $this->subscription->subscriptionPlan->name)
                    ->line('Crédits : ' . $this->subscription->credits_remaining)
                    ->line('Date de fin : ' . $subscriptionEndDate)
                    ->action('Voir mon abonnement', url("{$baseUrl}/student/subscription"));
                break;
            case 'finished':
                $message->line('Votre abonnement actuel est terminé.')
                    ->line('Plan : ' . $this->subscription->subscriptionPlan->name)
                    ->line('Crédits restants : 0')
                    ->line('Nous vous conseillons de souscrire à un nouvel abonnement dès que possible.')
                    ->action('Souscrire à un nouvel abonnement', url("{$baseUrl}/student/subscription"))
                    ->line('NB: Si vous avez activé le renouvellement automatique, votre abonnement sera renouvelé dans quelques instants.
                     Assurez-vous simplement d\'avoir le montant nécessaire.
                     Si le renouvellement n\'a pas lieu après 24 heures, veuillez souscrire à un nouvel abonnement manuellement.');
                break;
        }
        

        return $message->line('Merci d\'utiliser notre plateforme !');
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array
     */
    public function toArray($notifiable)
    {
        return [
            'subscription_id' => $this->subscription->id,
            'action' => $this->getActionUrl(),
            'content' => $this->getContent(),
            'subject' => $this->getSubject(),
            'image' => $this->getDefaultImage(),
            'initials' => $this->getInitial(),
        ];
    }

    public function broadcastOn()
    {
        return new Channel('users.'.$this->recieverId);
    }

    protected function getSubject()
    {
        switch ($this->action) {
            case 'activated':
                return 'Votre abonnement est activé';
            case 'subscription_with_lesson':
                return 'Nouvel abonnement étudiant - Leçon d\'essai requise';
            case 'subscription_expiring_soon':
                return 'Votre abonnement arrive bientôt à expiration';
            case 'subscription_auto_renewed':
                return 'Votre abonnement a été renouvelé';
            case 'subscription_auto_renewal_failed':
                return 'Échec du renouvellement automatique de votre abonnement';
            case 'subscription_expired':
                return 'Votre abonnement a expiré';
            case 'subscription_payment_failed':
                return 'Échec de paiement de l\'abonnement';
            case 'finished':
                return 'Votre abonnement est terminé';
            case 'canceled':
                return 'Votre abonnement a été annulé';
            case 'suspended':
                return 'Votre abonnement a été suspendu';
            case 'updated':
                return 'Votre abonnement a été mis à jour';
            case 'reactivation_after_suspension':
                return 'Votre abonnement a été réactivé après suspension';
            default:
                return 'Notification d\'abonnement';
        }
    }

    private function getActionUrl()
    {
        $baseUrl = env('FRONTEND_URL');

        switch ($this->action) {
            case 'activated':
            case 'subscription_expiring_soon':
            case 'subscription_auto_renewed':
            case 'subscription_expired':
            case 'subscription_auto_renewal_failed':
            case 'subscription_payment_failed':
            case 'canceled':
            case 'suspended':
            case 'updated':
            case 'finished':
            case 'reactivation_after_suspension':
                return [
                    'name' => 'student_subscription',
                    'fullUrl' => "{$baseUrl}/student/subscription"
                ];
            case 'subscription_with_lesson':
                return [
                    'name' => 'admin_lessons',
                    'fullUrl' => "{$baseUrl}/admin/lessons"
                ];
        }
    }

    protected function getContent()
    {
        $daysRemaining = now()->diffInDays($this->subscription->end_date);
        switch ($this->action) {
            case 'activated':
                return 'Votre abonnement a été activé avec succès!';
            case 'subscription_with_lesson':
                return 'l\'élève ' . $this->subscription->student->first_name . ' ' . $this->subscription->student->last_name . ' vient de souscrire à son premier abonnement et nécessite une leçon d\'essai';
            case 'subscription_expiring_soon':
                return "Votre abonnement arrive bientôt à expiration. Il vous reste {$daysRemaining} jour(s). Pensez à le renouveler.";
            case 'subscription_auto_renewed':
                return 'Votre abonnement a été renouvelé automatiquement.';
            case 'subscription_auto_renewal_failed':
                return 'Le renouvellement automatique de votre abonnement a échoué.';
            case 'subscription_expired':
                return 'Votre abonnement a expiré. Veuillez le renouveler.';
            case 'subscription_payment_failed':
                return 'Le paiement de votre abonnement a échoué. L\'abonnement a été annulé. Veuillez mettre à jour votre méthode de paiement et réessayer.';
            case 'canceled':
                return 'Votre abonnement a été manuellement annulé par un administrateur. ( Si vous pensez qu\'il s\'agit d\'une erreur, veuillez contacter le support )';    
            case 'suspended':
                return 'Votre abonnement a été suspendu par un administrateur. ( Si vous pensez qu\'il s\'agit d\'une erreur, veuillez contacter le support )';   
            case 'reactivation_after_suspension':
                return 'Votre abonnement a été réactivé après suspension.';    
            case 'updated':
                return 'Votre abonnement a été mis à jour.'; 
            case 'finished':
                return 'Votre abonnement actuel est terminé. Tous les crédits ont été utilisés.';  
            default:
                return 'Mise à jour de votre abonnement';
        }
    }

    private function getDefaultImage()
    {
        // Define default images for each action
        $images = [
            // 'created' => 'url_to_default_image_for_created.png',  // Replace with actual URL or path
            // 'updated' => 'url_to_default_image_for_updated.png',  // Replace with actual URL or path
            // 'deleted' => 'url_to_default_image_for_deleted.png',  // Replace with actual URL or path
        ];

        return $images[$this->action] ?? null;
    }

    private function getInitial()
    {
        // Define default images for each action
        $initials = [
            'activated' => 'AA',
            'subscription_with_lesson' => 'AL', 
            'subscription_expiring_soon' => 'ABE',
            'subscription_auto_renewed' => 'AR',
            'subscription_auto_renewal_failed' => 'RE',
            'subscription_expired' => 'AE',
            'canceled' => 'AAN',
            'suspended' => 'AS',
            'reactivation_after_suspension' => 'ARA',
            'updated' => 'AM',
            'finished' => 'AT',
            'subscription_payment_failed' => 'PE',
        ];

        return $initials[$this->action] ?? 'A'; 
    }
}