<?php

namespace App\Notifications;

use App\Models\User;
use App\Models\Lesson;
use Illuminate\Support\Str;
use Illuminate\Bus\Queueable;
use Illuminate\Broadcasting\Channel;
use Illuminate\Contracts\Broadcasting\ShouldBroadcast;
use Illuminate\Support\Facades\Auth;
use Illuminate\Notifications\Notification;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;

class ReportedProblemNotification extends Notification implements ShouldQueue, ShouldBroadcast
{
    use Queueable;

    protected $report;
    protected $recieverId;
    /**
     * Create a new notification instance.
     *
     * @return void
     */
    public function __construct($report, $recieverId)
    {
        $this->report = $report;
        $this->recieverId = $recieverId;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function via($notifiable)
    {
        $channels = [];
        return ['mail', 'database', 'broadcast'];

        return $channels;
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return \Illuminate\Notifications\Messages\MailMessage
     */
    public function toMail($notifiable)
    {
        $baseUrl = env('FRONTEND_URL');
        $reportDetail = $this->report->category == 'Autre' ? '' : ': '.$this->report->category;
        $reporter = User::find($this->report->reporter_id);
        $reporterName = $reporter ? ($reporter->hasRole('teacher') 
                                        ? $reporter->teacher->first_name.' '.$reporter->teacher->last_name 
                                        : $reporter->student->first_name.' '.$reporter->student->last_name) 
                                    : 'Utilisateur inconnu';
        
        $reportableInfo = $this->getReportableInfo();
    
        return (new MailMessage)
            ->subject('[URGENT] Nouveau signalement' . $reportDetail)
            ->line('Un nouveau signalement a été soumis sur la plateforme qui nécessite votre attention.')
            ->line('**Détails du signalement :**')
            ->line('• Motif du signalement : ' . $this->report->category)
            ->line('• Signalé par : ' . $reporterName)
            ->line('• Date : ' . $this->report->created_at->format('d/m/Y à H:i'))
            ->when(!empty($reportableInfo), function ($mail) use ($reportableInfo) {
                return $mail->line('• ' . $reportableInfo);
            })
            ->line('• Description : ' . $this->report->description)
            ->action('Examiner ce signalement', url($baseUrl.'/admin/reports/' . $this->report->id))
            ->line('Merci de traiter ce signalement et d\'agir en conséquence.');
    }
    
    /**
     * Get reportable information based on the type of reported entity.
     *
     * @return string
     */
    private function getReportableInfo()
    {
        if (!$this->report->reportable_id || !$this->report->reportable_type) {
            return '';
        }
    
        if ($this->report->reportable_type === 'User') {
            $user = User::find($this->report->reportable_id);
            if ($user) {
                return "Utilisateur signalé : " . ($user->hasRole('teacher') 
                    ? $user->teacher->first_name . ' ' . $user->teacher->last_name 
                    : $user->student->first_name . ' ' . $user->student->last_name);
            }
        } elseif ($this->report->reportable_type === 'Teacher_Lesson_Payment') {
            $lesson = Lesson::find($this->report->reportable_id);
            if ($lesson) {
                $lessonType = $lesson->lesson_type === 'trial' ? "d'essai" : "normale";
                return "Élément signalé : Rémunération de l'enseignant liée à la leçon {$lessonType} intitulée - " . $lesson->lesson_title;
            }
        }
    
        return "Élément signalé : Inconnu";
    }
    

    /**
     * Get the array representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function toArray($notifiable)
    {
        return [
            'report_id' => $this->report->id,
            'action' => $this->getActionUrl(),
            'content' => $this->getContent(),
            'subject' => 'Nouvelle réclamation soumise',
            'image' => $this->getDefaultImage(),
            'initials' => 'UR',
        ];
    }

    private function getActionUrl()
    {
        $baseUrl = env('FRONTEND_URL');

        return [
            'name' => 'admin_reportDetail',
            'params' => ['rId' => $this->report->id],
            'fullUrl' => "{$baseUrl}/admin/reports/{$this->report->id}",
        ];
    }

    private function getContent()
    {
        $reason = $this->report->category == 'Autre' ? 'raison inconnu' : $this->report->category;
        return "[URGENT] Un nouveau signalement a été soumis pour **". Str::title($reason) . "**";
    }

    public function broadcastOn()
    {
        return new Channel('users.'.$this->recieverId);
    }

    private function getDefaultImage()
    {
        // Define default images for each action
        $images = [
            // 'created' => 'url_to_default_image_for_created.png',  // Replace with actual URL or path
            // 'updated' => 'url_to_default_image_for_updated.png',  // Replace with actual URL or path
            // 'deleted' => 'url_to_default_image_for_deleted.png',  // Replace with actual URL or path
        ];

        return null; // Fallback to a default image
    }
}
