<?php

namespace App\Notifications;

use Carbon\Carbon;
use App\Models\User;
use App\Models\Lesson;
use Illuminate\Bus\Queueable;
use Illuminate\Broadcasting\Channel;
use Illuminate\Notifications\Notification;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Contracts\Broadcasting\ShouldBroadcast;

class LessonNotification extends Notification implements ShouldQueue, ShouldBroadcast
{
    use Queueable;

    public $lesson;
    public $action;
    public $recipientType;
    public $recieverId; 

    /**
     * Create a new notification instance.
     */
    public function __construct(Lesson $lesson, string $action, string $recipientType, $recieverId)
    {
        $this->lesson = $lesson;
        $this->action = $action;
        $this->recipientType = $recipientType;
        $this->recieverId = $recieverId;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array
     */
    public function via($notifiable)
    {
        $channels = [];

        if ($notifiable->notificationPref->in_app_notifications) {
            $channels[] = 'database';
            $channels[] = 'broadcast';
        }

        if (($notifiable->notificationPref->email_notifications 
            && !in_array($this->action, ['credit_returned_active', 'credit_returned_passed']))
            || in_array($this->action, ['created', 'trial_lesson_created','credit_not_return','upcoming','reprogrammed'])) {
            $channels[] = 'mail';
        }

        return $channels;
    }

    /**
     * Get the mail representation of the notification.
     *
     * @return MailMessage
     */
    public function toMail($notifiable)
    {
        $message = (new MailMessage())->subject($this->getSubject());
        $reciver = User::find($this->recieverId);
        $isTrialLesson = $this->lesson->lesson_type === 'trial';
        $userDateTime = Carbon::parse($this->lesson->date . ' ' . $this->lesson->time)
                    ->timezone($reciver->timezone)
                    ->format('d-m-Y, H:i');

        switch ($this->action) {
            case 'created':
                $message->line('Nouvelle leçon')
                        ->line('Type de leçon : '.($isTrialLesson ? "Leçon d'essai" : "Leçon normale"))
                        ->line('Titre de leçon : '.$this->lesson->lesson_title)
                        ->line($this->recipientType === 'teacher' 
                                ? 'Élève : '.$this->lesson->student->first_name . ' ' . $this->lesson->student->last_name
                                : 'Enseignant : M/Mme. '.$this->lesson->teacher->first_name . ' ' . $this->lesson->teacher->last_name)
                        ->line('Matière : '.$this->lesson->course->course_title)
                        ->line('Date programmée : '.$userDateTime)
                        ->action('Voir la leçon', url("{$this->getActionUrl('email')}"));
                break;

            case 'trial_lesson_created':
                $message->line('Une nouvelle demande de leçon d\'essai a été créée.')
                    ->line('Élève : '.$this->lesson->student->first_name . ' ' . $this->lesson->student->last_name)
                    ->line('Titre de leçon : '.$this->lesson->lesson_title)
                    ->line('Matière : '.$this->lesson->course->course_title)
                    ->line('Date proposée : '.$userDateTime)
                    ->action('Voir la plus', url("{$this->getActionUrl('email')}"));
                break;    

            case 'lesson_request':
                $message->line('Nouvelle demande de leçon en attente de confirmation.')
                        ->line('Type de leçon : '.($isTrialLesson ? "Leçon d'essai" : "Leçon normale"))
                        ->line('Titre de leçon : '.$this->lesson->lesson_title)
                        ->line('Élève : '.$this->lesson->student->first_name . ' ' . $this->lesson->student->last_name)
                        ->line('Matière : '.$this->lesson->course->course_title)
                        ->line('Date programmée : '.$userDateTime)
                        ->action('Voir la leçon', url("{$this->getActionUrl('email')}"));
                break;

            case 'lesson_request_pending':
                $message->line("Une nouvelle leçon a été créée par un administrateur et est en attente d'approbation par l'enseignant.")
                        ->line('Type de leçon : '.($isTrialLesson ? "Leçon d'essai" : "Leçon normale"))
                        ->line('Titre de leçon : '.$this->lesson->lesson_title)
                        ->line('Enseignant : M/Mme. '.$this->lesson->teacher->first_name . ' ' . $this->lesson->teacher->last_name)
                        ->line('Matière : '.$this->lesson->course->course_title)
                        ->line('Date programmée : '.$userDateTime)
                        ->action('Voir la leçon', url("{$this->getActionUrl('email')}"));
                break;
        
            case 'accepted':
                $message->line($this->recipientType === 'teacher' ? 'Leçon confirmée.' : 'Leçon confirmée par un administrateur.')
                        ->line('Type de leçon : '.($isTrialLesson ? "Leçon d'essai" : "Leçon normale"))
                        ->line('Titre de leçon : '.$this->lesson->lesson_title)
                        ->line($this->recipientType === 'teacher' 
                                ? 'Élève : '.$this->lesson->student->first_name . ' ' . $this->lesson->student->last_name
                                : 'Enseignant : M/Mme. '.$this->lesson->teacher->first_name . ' ' . $this->lesson->teacher->last_name)
                        ->line('Matière : '.$this->lesson->course->course_title)
                        ->line('Date programmée : '.$userDateTime)
                        ->action('Voir la leçon', url("{$this->getActionUrl('email')}"));
                break;
        
            case 'rejected':
                $message->line('Leçon rejetée')
                        ->line('Type de leçon : '.($isTrialLesson ? "Leçon d'essai" : "Leçon normale"))
                        ->line('Titre de leçon : '.$this->lesson->lesson_title)
                        ->line('Enseignant : M/Mme. '.$this->lesson->teacher->first_name . ' ' . $this->lesson->teacher->last_name)
                        ->line('Matière : '.$this->lesson->course->course_title)
                        ->line('Date programmée : '.$userDateTime);

                if ($isTrialLesson && $this->recipientType === 'admin') {
                    $message->line('Élève : '.$this->lesson->student->first_name . ' ' . $this->lesson->student->last_name)
                            ->line('NB: Une nouvelle leçon d\'essai doit être reprogrammée pour l\'élève.');
                } elseif (!$isTrialLesson && $this->recipientType === 'student') {
                    $message->line('NB: Si un crédit a été consommé pour cette leçon, il sera retourné à votre abonnement actuel. 
                    Si vous n\'avez pas d\'abonnement actif ou suspendu, l\'un de vos abonnements précédents sera réactivé avec le crédit envoyé à celui-ci. 
                    Si aucun abonnement ne peut être trouvé, un message sera envoyé aux administrateurs.');
                }

                $message->action('Voir la leçon', url("{$this->getActionUrl('email')}"));
                break;
        
            case 'updated':
                $message->line('Leçon modifiée.')
                        ->line('Type de leçon : '.($isTrialLesson ? "Leçon d'essai" : "Leçon normale"))
                        ->line('Titre de leçon : '.$this->lesson->lesson_title)
                        ->line('Matière : '.$this->lesson->course->course_title)
                        ->line('Date programmée : '.$userDateTime)
                        ->action('Voir la leçon', url("{$this->getActionUrl('email')}"));
                break;
        
            case 'deleted':
                $message->line('Leçon supprimée.')
                        ->line('Type de leçon : '.($isTrialLesson ? "Leçon d'essai" : "Leçon normale"))
                        ->line('Titre de leçon : '.$this->lesson->lesson_title)
                        ->line('Matière : '.$this->lesson->course->course_title);
                break;
            
            case 'canceled':
                $message->line('Leçon annulée')
                        ->line('Type de leçon : '.($isTrialLesson ? "Leçon d'essai" : "Leçon normale"))
                        ->line('Titre de leçon : '.$this->lesson->lesson_title)
                        ->line($this->recipientType === 'teacher' 
                            ? 'Élève : '.$this->lesson->student->first_name . ' ' . $this->lesson->student->last_name
                            : 'Enseignant : M/Mme. '.$this->lesson->teacher->first_name . ' ' . $this->lesson->teacher->last_name)
                        ->line('Matière : '.$this->lesson->course->course_title);

                if ($isTrialLesson && $this->recipientType === 'admin') {
                    $message->line('Élève : '.$this->lesson->student->first_name. ' ' . $this->lesson->student->last_name)
                            ->line('NB: Une nouvelle leçon d\'essai doit être reprogrammée pour l\'élève.');
                } elseif (!$isTrialLesson && $this->recipientType === 'student') {
                    $message->line('NB: Si un crédit a été consommé pour cette leçon, il sera retourné à votre abonnement actuel. 
                    Si vous n\'avez pas d\'abonnement actif ou suspendu, l\'un de vos abonnements précédents sera réactivé avec le crédit envoyé à celui-ci. 
                    Si aucun abonnement ne peut être trouvé, un message sera envoyé aux administrateurs.');
                }

                $message->action('Voir la leçon', url("{$this->getActionUrl('email')}"));
                break;

            case 'completed':
                $message->line('Leçon terminée.')
                        ->line('Type de leçon : '.($isTrialLesson ? "Leçon d'essai" : "Leçon normale"))
                        ->line('Titre de leçon : '.$this->lesson->lesson_title)
                        ->line('Matière : '.$this->lesson->course->course_title)
                        ->action('Voir les détails', url("{$this->getActionUrl('email')}"));
                break;

            case 'expired':
                $message->line('Leçon expirée.')
                        ->line('Type de leçon : '.($isTrialLesson ? "Leçon d'essai" : "Leçon normale"))
                        ->line('Titre de leçon : '.$this->lesson->lesson_title)
                        ->line('Matière : '.$this->lesson->course->course_title)
                        ->line('Date programmée : '.$userDateTime)
                        ->action('Voir les détails', url("{$this->getActionUrl('email')}"));
                break;

            case 'reprogrammed':
                $message->line('Leçon reprogrammée.')
                        ->line('Type de leçon : '.($isTrialLesson ? "Leçon d'essai" : "Leçon normale"))
                        ->line('Titre de leçon : '.$this->lesson->lesson_title)
                        ->line('Matière : '.$this->lesson->course->course_title)
                        ->line('Nouvelle date : '.$userDateTime)
                        ->action('Voir les détails', url("{$this->getActionUrl('email')}"));
                break;                

            case 'upcoming':
                $message->line('Rappel : Vous avez une leçon à venir.')
                        ->line('Type de leçon : '.($isTrialLesson ? "Leçon d'essai" : "Leçon normale"))
                        ->line('Titre de leçon : '.$this->lesson->lesson_title)
                        ->line('Matière : '.$this->lesson->course->course_title)
                        ->line('Date programmée : '.$userDateTime)
                        ->action('Voir la leçon', url("{$this->getActionUrl('email')}"));
                break;
            case 'credit_not_return':
                $message->line('[URGENT] Échec de retour de crédit')
                        ->line('Type de leçon : '.($isTrialLesson ? "Leçon d'essai" : "Leçon normale"))
                        ->line('Titre de leçon : '.$this->lesson->lesson_title)
                        ->line('Matière : '.$this->lesson->course->course_title)
                        ->line('Date programmée : '.$userDateTime)
                        ->line('Enseignant : M/Mme. '.$this->lesson->teacher->first_name . ' ' . $this->lesson->teacher->last_name);
                        if ($this->recipientType === 'admin') {
                            $message->line('Élève concerné : '.$this->lesson->student->first_name . ' ' . $this->lesson->student->last_name)
                                    ->line('Statut : Le crédit n\'a pas pu être retourné à un abonnement après l\'annulation ou le rejet de la leçon.');
                        } else {
                            $message->line('Statut : Le crédit n\'a pas pu être retourné à un abonnement après l\'annulation ou le rejet de la leçon. Les administrateurs ont été notifiés de cela.');
                        }
                        $message->action('Voir la leçon', url("{$this->getActionUrl('email')}"));
                break;
        }
        

        return $message->line('Merci d\'utiliser notre application !');
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array
     */
    public function toArray($notifiable)
    {
        return [
            'lesson_id' => $this->lesson->id,
            'action' => $this->getActionUrl('database'),
            'content' => $this->getContent(),
            'subject' => $this->getSubject(),
            'image' => $this->getDefaultImage(),
            'initials' => $this->getInitial(),
        ];
    }

    public function broadcastOn()
    {
        return new Channel('users.'.$this->recieverId);
    }

    private function getSubject()
    {
        switch ($this->action) {
            case 'created':
                return 'Nouvelle leçon programmée';
            case 'updated':
                return 'Leçon mise à jour';
            case 'reprogrammed':
                return 'Leçon reprogrammée';
            case 'deleted':
                return 'Leçon supprimée';
            case 'canceled':
                return 'Leçon annulée';
            case 'completed':
                return 'Leçon terminée';
            case 'expired':
                return 'Leçon expirée';
            case 'upcoming':
                return 'Rappel : Leçon à venir';
            case 'trial_lesson_created':
                return 'Nouvelle demande de leçon d\'essai';
            case 'lesson_request':
                return 'Leçon en attente de confirmation';
            case 'lesson_request_pending':
                return 'Leçon en attente de confirmation';
            case 'accepted':
                return 'Leçon acceptée';
            case 'rejected':
                return 'Leçon rejetée';
            case 'credit_not_return':
                return '[URGENT] Échec de retour de crédit';
            case 'credit_returned_passed':
                return 'Crédit retourné à un abonnement passé';
            case 'credit_returned_active':
                return 'Crédit retourné à l\'abonnement actuel';
            default:
                return 'Notification de leçon';
        }
    }

    private function getContent()
    {
        $isTrialLesson = $this->lesson->lesson_type === 'trial';
        $reciver = User::find($this->recieverId);
        $lessonType = $isTrialLesson ? "d'essai" : "normale";
        $userDateTime = Carbon::parse($this->lesson->date . ' ' . $this->lesson->time)
                            ->timezone($reciver->timezone)
                            ->format('d-m-Y, H:i');
                            
        switch ($this->action) {
            case 'created':
                return $this->recipientType === 'student' 
                        ? "Nouvelle leçon {$lessonType} intitulée \"{$this->lesson->lesson_title}\" programmée avec M/Mme. {$this->lesson->teacher->first_name} {$this->lesson->teacher->last_name} pour le " . $userDateTime 
                        : "Nouvelle leçon {$lessonType} intitulée \"{$this->lesson->lesson_title}\" programmée avec {$this->lesson->student->first_name} {$this->lesson->student->last_name} pour le " . $userDateTime;

            case 'trial_lesson_created':
                return "Une nouvelle demande de leçon d'essai a été créée par \"{$this->lesson->student->first_name} {$this->lesson->student->last_name}\".";
            
            case 'lesson_request':
                return "Une nouvelle leçon {$lessonType} intitulée \"{$this->lesson->lesson_title}\" est en attente de confirmation.";

            case 'lesson_request_pending':
                return "Une nouvelle leçon {$lessonType} intitulée \"{$this->lesson->lesson_title}\" a été créée par un administrateur et est en attente d'approbation par M/Mme. {$this->lesson->teacher->first_name} {$this->lesson->teacher->last_name}.";
        
            case 'accepted':
                return $this->recipientType === 'student' 
                        ? "Votre leçon {$lessonType} intitulée \"{$this->lesson->lesson_title}\" avec M/Mme. {$this->lesson->teacher->first_name} {$this->lesson->teacher->last_name} est confirmée pour le " . $userDateTime 
                        : "Votre leçon {$lessonType} intitulée \"{$this->lesson->lesson_title}\" avec {$this->lesson->student->first_name} {$this->lesson->student->last_name} a été confirmée par un administrateur pour le " . $userDateTime;

            case 'rejected':
                return $isTrialLesson && $this->recipientType === 'admin'
                    ? "La leçon d'essai de {$this->lesson->student->first_name} {$this->lesson->student->last_name} a été rejetée par M/Mme. {$this->lesson->teacher->first_name} {$this->lesson->teacher->last_name}.Une nouvelle leçon d'essai doit être programmée pour l'élève." 
                    : ( $this->recipientType === 'admin' 
                        ? "La leçon {$lessonType} intitulée \"{$this->lesson->lesson_title}\" que vous avez programmée a été rejetée par M/Mme. {$this->lesson->teacher->first_name} {$this->lesson->teacher->last_name}."
                        : "Votre leçon {$lessonType} intitulée \"{$this->lesson->lesson_title}\" a été rejetée par M/Mme. {$this->lesson->teacher->first_name} {$this->lesson->teacher->last_name}.");
        
            case 'updated':
                return "Votre leçon {$lessonType} intitulée \"{$this->lesson->lesson_title}\" a été mise à jour.";

            case 'reprogrammed':
                return "Votre leçon {$lessonType} intitulée \"{$this->lesson->lesson_title}\" a été reprogrammée pour le " . $userDateTime;
        
            case 'deleted':
                return "Votre leçon {$lessonType} intitulée \"{$this->lesson->lesson_title}\" a été supprimée.";

            case 'canceled':
                return $isTrialLesson && $this->recipientType === 'admin'
                    ? "La leçon d'essai de {$this->lesson->student->first_name} {$this->lesson->student->last_name} a été annulée. Une nouvelle leçon d'essai doit être programmée pour l'élève." 
                    : ($this->recipientType === 'student' 
                    ? "Votre leçon {$lessonType} intitulée \"{$this->lesson->lesson_title}\" avec M/Mme. {$this->lesson->teacher->first_name} {$this->lesson->teacher->last_name} a été annulée." 
                    : "Votre leçon {$lessonType} intitulée \"{$this->lesson->lesson_title}\" avec {$this->lesson->student->first_name} {$this->lesson->student->last_name} a été annulée.");
        
            case 'completed':
                return "Votre leçon {$lessonType} intitulée \"{$this->lesson->lesson_title}\" est désormais terminée.";

            case 'expired':
                return "Votre leçon {$lessonType} intitulée \"{$this->lesson->lesson_title}\" est désormais expirée.";

            case 'credit_not_return':
                return $this->recipientType === 'student'
                    ? "[URGENT] Le crédit pour votre leçon {$lessonType} intitulée \"{$this->lesson->lesson_title}\" n'a pas pu être retourné à un abonnement après l\'annulation ou le rejet de la leçon. Les administrateurs ont été notifiés de cela."
                    : "[URGENT] Le crédit pour la leçon {$lessonType} intitulée \"{$this->lesson->lesson_title}\" de {$this->lesson->student->first_name} {$this->lesson->student->last_name} n'a pas pu être retourné à un abonnement après l\'annulation ou le rejet de la leçon.";

            case 'credit_returned_passed':
                return "Votre crédit pour la leçon {$lessonType} intitulée \"{$this->lesson->lesson_title}\" a été retourné en activant un abonnement passé suite à un rejet ou une annulation de la leçon.";

            case 'credit_returned_active':
                return "Votre crédit pour la leçon {$lessonType} intitulée \"{$this->lesson->lesson_title}\" a été retourné à votre abonnement actuel suite à un rejet ou une annulation de la leçon.";

            case 'upcoming':
                return "Rappel : Vous avez une leçon {$lessonType} intitulée \"{$this->lesson->lesson_title}\" à venir le " . $userDateTime;
        
            default:
                return "Notification concernant la leçon intitulée \"{$this->lesson->lesson_title}\".";
        }
        
    }

    private function getActionUrl($via)
    {
        $baseUrl = env('FRONTEND_URL');

        if ($via === 'database') {
            if (in_array($this->action, ['credit_returned_active', 'credit_returned_passed', 'credit_not_return'])) {
                return [
                    'name' => $this->recipientType === 'admin' ? 'admin_subscriptions' : 'student_subscription',
                    'fullUrl' => "{$baseUrl}/" . ($this->recipientType === 'admin' ? 'admin/subscriptions' : 'student/subscription')
                ];
            } else {
                return [
                    'name' => $this->recipientType.'_lessonDetail',
                    'params' => ['lId' => $this->lesson->id],
                    'fullUrl' => "{$baseUrl}/{$this->recipientType}/lessons/{$this->lesson->id}"
                ];
            }
        } else {
            return "{$baseUrl}/{$this->recipientType}/lessons/{$this->lesson->id}";
        }
    }

    private function getDefaultImage()
    {
        // Define default images for each action
        $images = [
            // 'created' => 'url_to_default_image_for_created.png',  // Replace with actual URL or path
            // 'updated' => 'url_to_default_image_for_updated.png',  // Replace with actual URL or path
            // 'deleted' => 'url_to_default_image_for_deleted.png',  // Replace with actual URL or path
        ];

        return $images[$this->action] ?? null; // Fallback to a default image
    }

    private function getInitial()
    {
        // Define default initails for each action
        $initails = [
            'created' => 'LP',  // Replace with actual URL or path
            'updated' => 'LM',  // Replace with actual URL or path
            'deleted' => 'LS',  // Replace with actual URL or path
            'trial_lesson_created' => 'LE',
            'lesson_request' => 'LA',
            'lesson_request_pending' => 'LA',
            'accepted' => 'LC',
            'rejected' => 'LR',
            'canceled' => 'LA',
            'expired' => 'LE',
            'upcoming' => 'LA',
            'completed' => 'LT',
            'reprogrammed' => 'LR',
        ];

        return $initails[$this->action] ?? 'L'; // Fallback to a default image
    }
}