<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Notifications\Notification;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;

class AccountDeletedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    protected $deletedBy;
    protected $isSoftDelete;

    /**
     * Create a new notification instance.
     *
     * @param string $deletedBy 'self' or 'admin'
     * @param bool $isSoftDelete
     * @param string|null $adminName
     * @return void
     */
    public function __construct($deletedBy = 'self', $isSoftDelete = false)
    {
        $this->deletedBy = $deletedBy;
        $this->isSoftDelete = $isSoftDelete;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function via($notifiable)
    {
        return ['mail'];
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return \Illuminate\Notifications\Messages\MailMessage
     */
    public function toMail($notifiable)
    {

        $mailMessage = (new MailMessage)
            ->subject('Suppression de votre compte')
            ->greeting('Bonjour,');

        if ($this->deletedBy === 'self') {
            $mailMessage
                ->line('Votre compte a été supprimé à votre demande.')
                ->line('Si vous n\'êtes pas à l\'origine de cette action, veuillez contacter immédiatement notre support administratif pour plus de détails.')
                ->line('Email du support : ' . env('COMPANY_EMAIL'));
        } else {
            if ($this->isSoftDelete) {
                $mailMessage
                    ->line('Votre compte a été partiellement supprimé par les administrateurs de la plateforme.')
                    ->line('Certaines de vos informations ont été conservées. Pour restaurer votre compte :')
                    ->line('1. Connectez-vous à votre compte')
                    ->line('2. Suivez les étapes de restauration qui vous seront présentées')
                    ->line('Si vous avez des questions sur cette procédure, notre équipe support est à votre disposition.')
                    ->action('Commencer la restauration', env('FRONTEND_URL') . '/login');
            } else {
                $mailMessage
                    ->line('Votre compte a été supprimé définitivement par les administrateurs de la plateforme.')
                    ->line('Pour plus d\'informations sur cette action, vous pouvez contacter notre équipe support à l\'adresse : ' . env('COMPANY_EMAIL'));
            }
        }

        $mailMessage->line('Merci d\'avoir utilisé notre plateforme.');
        
        return $mailMessage;
    }
}