<?php

namespace App\Jobs;

use Stripe\Stripe;
use App\Models\Student;
use App\Events\UpdateCredit;
use App\Models\Subscription;
use Illuminate\Bus\Queueable;
use Illuminate\Support\Facades\Log;
use Illuminate\Queue\SerializesModels;
use App\Http\Controllers\UserController;
use Illuminate\Queue\InteractsWithQueue;
use App\Http\Controllers\PaymentController;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use App\Notifications\SubscriptionNotification;

class ProcessAutoRenewalJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $subscription;
    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(Subscription $subscription)
    {
        $this->subscription = $subscription;
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle()
    {

        log::info('Auto Renewal Job Started');
        try {
            // Retrieve associated student and subscription plan
            $student = Student::find($this->subscription->student_id);
            $subscriptionPlan = $this->subscription->subscriptionPlan;
            $user = $student->user;
            $paymentMethods = $user->paymentMethods()->where('status', 'active')->get();

            Stripe::setApiKey(env('STRIPE_SECRET'));
            $customerId = (new UserController())->createStripeCustomerAccount($user)->id;
            $stripeCustomer = \Stripe\Customer::retrieve($customerId);
            $defaultPaymentMethodId = $stripeCustomer->invoice_settings->default_payment_method;

            if ($defaultPaymentMethodId) {
                $paymentMethods = $paymentMethods->sortBy(function ($method) use ($defaultPaymentMethodId) {
                    return $method->stripe_payment_method_id === $defaultPaymentMethodId ? 0 : 1;
                });
            }

            foreach ($paymentMethods as $paymentMethod) {
                try {
                    $paymentData = [
                        'payment_method_id' => $paymentMethod->stripe_payment_method_id,
                        'amount' => $subscriptionPlan->price * 100,
                        'student_id' => $student->id,
                        'subscription_plan_id' => $subscriptionPlan->id,
                        'free_lesson' => false
                    ];

                    $request = new \Illuminate\Http\Request();
                    $request->replace($paymentData);

                    $paymentController = app(PaymentController::class);
                    $response = $paymentController->SaveCardPayment($request);
                    $responseData = $response->getData();

                    if ($responseData->status === 'success') {
                        $user->notify(new SubscriptionNotification($responseData->subscription, 'subscription_auto_renewed', $user->id));
                        event(new UpdateCredit($user->id));
                        Log::info("Successfully auto-renewed Subscription ID {$this->subscription->id}.");
                        return; // Stop processing after successful renewal
                    } elseif ($responseData->status === 'processing') {
                        Log::info("Auto-renewal processing for Subscription ID {$this->subscription->id}.");
                        return; // Consider processing as successful
                    }

                    // If not successful, try the next payment method
                    Log::info("Auto-renewal failed with payment method {$paymentMethod->id}. Trying next method.");
                } catch (\Exception $e) {
                    Log::info("Error processing payment method {$paymentMethod->id}: " . $e->getMessage());
                    // Continue to the next payment method
                }
            }

            // If all payment methods fail
            $user->notify(new SubscriptionNotification($this->subscription, 'subscription_auto_renewal_failed', $user->id));
            Log::info("All payment methods failed for Subscription ID {$this->subscription->id}.");
        } catch (\Exception $e) {
            Log::error("Auto-renewal error for Subscription ID {$this->subscription->id}: " . $e->getMessage());
            $user->notify(new SubscriptionNotification($this->subscription, 'subscription_auto_renewal_failed', $user->id));
        }
    }
}
