<?php

namespace App\Jobs;

use App\Models\Meeting;
use Carbon\Carbon;
use GuzzleHttp\Client;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class MonitorMeetingDurations implements ShouldQueue
{
    use Dispatchable;
    use InteractsWithQueue;
    use Queueable;
    use SerializesModels;

    public function handle()
    {
        $apiKey = env('WHEREBY_API_KEY');
        $client = new Client();
        $maxDuration = 5; // Maximum meeting duration in minutes

        $activeMeetings = Meeting::whereHas('lesson', function ($query) {
            $query->where('status', 'in_progress');
        })->get();

        foreach ($activeMeetings as $meeting) {
            try {
                $response = $client->get("https://api.whereby.dev/v1/insights/room-sessions?roomName={$meeting->roomName}", [
                    'headers' => [
                        'Authorization' => 'Bearer '.$apiKey,
                    ],
                ]);

                $sessionInsights = json_decode($response->getBody(), true);

                if ($response->getStatusCode() === 200 && !empty($sessionInsights['results'])) {
                    foreach ($sessionInsights['results'] as $session) {
                        if ($session['roomName'] === $meeting->roomName) {
                            $startedAt = Carbon::parse($session['startedAt']);
                            $currentTime = now();
                            $duration = $currentTime->diffInMinutes($startedAt);

                            if ($duration >= $maxDuration) {
                                $this->endMeeting($meeting->room_id, $client, $apiKey);
                                $meeting->status = 'ended';
                                $meeting->save();
                            }
                        }
                    }
                } else {
                    logger()->error('Failed to retrieve session insights for meeting '.$meeting->id.'. Status code: '.$response->getStatusCode());
                }
            } catch (\Exception $e) {
                logger()->error('Error retrieving session insights for meeting '.$meeting->id.': '.$e->getMessage());
            }
        }

        // Dispatch the job again to run periodically
        self::dispatch()->delay(now()->addMinute());
    }

    private function endMeeting($roomId, $client, $apiKey)
    {
        try {
            $response = $client->delete("https://api.whereby.dev/v1/meetings/{$roomId}", [
                'headers' => [
                    'Authorization' => 'Bearer '.$apiKey,
                ],
            ]);

            if ($response->getStatusCode() === 204) {
                $meeting = Meeting::where('room_id', $roomId)->first();
                if ($meeting) {
                    $meeting->lesson->status = 'completed';
                    $meeting->lesson->save();
                    logger()->info("Lesson status for meeting with room ID {$roomId} set to 'completed'.");
                } else {
                    logger()->warning("Failed to find lesson associated with meeting room ID {$roomId}.");
                }
            } else {
                logger()->error("Failed to end meeting with room ID {$roomId} due to duration limit. Status code: {$response->getStatusCode()}");
            }
        } catch (\Exception $e) {
            logger()->error("Error ending meeting with room ID {$roomId} due to duration limit: {$e->getMessage()}");
        }
    }
}
