<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use Stripe\Stripe;
use Stripe\Webhook;
use App\Models\User;
use App\Models\Payment;
use App\Models\Student;
use Stripe\PaymentIntent;
use App\Events\UpdateCredit;
use App\Models\Subscription;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Log;
use App\Notifications\SubscriptionNotification;

class WebhookController extends Controller
{
    public function handleStripeWebhook(Request $request)
    {
        Stripe::setApiKey(env('STRIPE_SECRET'));

        $endpointSecret = env('STRIPE_WEBHOOK_SECRET'); // Set your webhook secret in the .env file
        $payload = $request->getContent();
        $sigHeader = $request->header('Stripe-Signature');
        $event = null;

        try {
            $event = Webhook::constructEvent(
                $payload,
                $sigHeader,
                $endpointSecret
            );
        } catch (\UnexpectedValueException $e) {
            // Invalid payload
            return response()->json(['error' => 'Invalid payload'], 400);
        } catch (\Stripe\Exception\SignatureVerificationException $e) {
            // Invalid signature
            return response()->json(['error' => 'Invalid signature'], 400);
        }

        // Handle different types of events
        switch ($event->type) {
            case 'payment_intent.succeeded':
                $intent = $event->data->object;
                $this->handlePaymentSucceeded($intent);
                break;

            case 'payment_intent.payment_failed':
                $intent = $event->data->object;
                $this->handlePaymentFailed($intent);
                break;
                
            case 'payment_intent.processing':
                $intent = $event->data->object;
                PaymentIntent::update($intent->id, [
                    'metadata' => [
                        'status_to_handle' => 'processing', // Example metadata key
                    ],
                ]);
                break;

            default:
                // Log unhandled event types
                Log::info('Unhandled event type: ' . $event->type);
        }

        return response()->json(['status' => 'success'], 200);
    }

    private function handlePaymentSucceeded($intent)
    {
        // Example: Update your subscription and payment status in the database
        Log::info("Payment Succeeded: " . $intent->id);

        // Use metadata to link payment to a subscription
        $payment = Payment::where('stripe_payment_id',$intent->id)->where('status', 'pending')->first();
        $subscription = $payment->subscription;
        $statusToHandle = $intent->metadata->status_to_handle ?? null;

        if ($payment && $subscription && $statusToHandle === 'processing') {
            $payment->status = 'completed';
            $payment->stripe_payment_id = $intent->id;
            $payment->stripe_payment_method_id = $intent->payment_method;
            $payment->save();

            $existingSubscription = Subscription::where('student_id', $subscription->student->id)->where('status', 'active')->orderBy('created_at', 'desc')->first();
            $remainingDuration = 0;
            $creditsRemaining = 0;

            if ($existingSubscription) {
                $remainingDuration = Carbon::parse($existingSubscription->end_date)->diffInDays(Carbon::now('UTC'));
                $creditsRemaining = $existingSubscription->credits_remaining;
                $existingSubscription->update([
                    'status' => 'finished',
                    'credits_remaining' => 0,
                    'end_date' => Carbon::now('UTC')->toDateTimeString(),
                ]);
            }

            $subscription->status = 'active';
            $subscription->start_date = Carbon::now('UTC')->toDateTimeString();
            $subscription->end_date = Carbon::now('UTC')->addDays($subscription->subscriptionPlan->duration_days + $remainingDuration);
            $subscription->credits_remaining = ($subscription->credits_remaining ?: 0) + $creditsRemaining;
            $subscription->save();
            $student = Student::find($subscription->student_id);
            $student->user->notify(new SubscriptionNotification($subscription, 'activated', $student->user->id));
            event(new UpdateCredit($student->user->id));

            if($subscription->includes_free_lesson){
                $admins = User::role('admin')->get();
                foreach ($admins as $admin) {
                    $admin->notify(new SubscriptionNotification($subscription, 'subscription_with_lesson', $admin->id));
                }
            }
        }
    }

    private function handlePaymentFailed($intent)
    {
        // Example: Update the subscription status to failed
        Log::error("Payment Failed: " . $intent->id);

        // Use metadata to link payment to a subscription
        $payment = Payment::where('stripe_payment_id',$intent->id)->where('status', 'pending')->first();
        $subscription = $payment->subscription;
        $statusToHandle = $intent->metadata->status_to_handle ?? null;

        if ($payment && $subscription && $statusToHandle === 'processing') {
            $payment->status = 'failed';
            $payment->stripe_payment_id = $intent->id;
            $payment->stripe_payment_method_id = $intent->payment_method;
            $payment->save();

            $subscription->start_date = Carbon::now('UTC')->toDateTimeString();
            $subscription->end_date = Carbon::now('UTC')->toDateTimeString();
            $subscription->status = 'canceled';
            $subscription->payment_method_id = null;
            $subscription->credits_remaining = 0;
            $subscription->save();
            
            $student = Student::find($subscription->student_id);
            $student->user->notify(new SubscriptionNotification($subscription, 'subscription_payment_failed', $student->user->id));

        }

        $errorMessage = $intent->last_payment_error
            ? $intent->last_payment_error->message
            : 'Unknown error';
        Log::error("Payment Error Message: " . $errorMessage);
    }
}
