<?php

namespace App\Http\Controllers;

use Pusher\Pusher;
use Stripe\Stripe;
use App\Models\User;
use Stripe\Customer;
use App\Models\Lesson;
use App\Models\Message;
use App\Models\Student;
use App\Models\Teacher;
use Stripe\PaymentMethod;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Events\AllMessagesRead;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use App\Mail\EmailUpdateConfirmation;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Password;
use App\Notifications\LessonNotification;
use App\Notifications\ResetPasswordNotification;
use App\Notifications\AccountDeletedNotification;
use App\Notifications\AccountRestoreNotification;

class UserController extends Controller
{
    public function index()
    {
        $users = User::orderBy('updated_at', 'desc')->get();

        return response()->json(['users' => $users]);
    }

    public function getContact(User $user)
    {
        $query = User::query();

        if ($user->hasRole('teacher')) {
            $query->whereHas('student.lessons', function ($q) use ($user) {
                $q->where('teacher_id', $user->teacher->id);
            })->role('student');
        } elseif ($user->hasRole('student')) {
            $query->whereHas('teacher.lessons', function ($q) use ($user) {
                $q->where('student_id', $user->student->id);
            })->role('teacher');
        } elseif ($user->hasRole('admin')) {
            $query->role(['student', 'teacher']);
        } else {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        // Add admins to the query if the user is a teacher or student
        if ($user->hasRole('teacher') || $user->hasRole('student')) {
            $adminUsers = User::role('admin')->select('id', 'email')->get();
        }

        $users = $query->withTrashed()->select('id', 'email', 'status')->with(['student:user_id,first_name,last_name,profile_image', 'teacher:user_id,first_name,last_name,profile_image'])->get();

        if (isset($adminUsers)) {
            $users = $users->merge($adminUsers);
        }

        $formattedUsers = $users->map(function ($user) {
            $data = [
                'user_id' => $user->id,
                'status' => $user->status,
            ];

            if ($user->student) {
                $data['name'] = $user->student->first_name.' '.$user->student->last_name;
                $data['role'] = 'student';
                $data['profile_image'] = $user->student->profile_image ? asset('storage/'.$user->student->profile_image) : null;
            } elseif ($user->teacher) {
                $data['name'] = $user->teacher->first_name.' '.$user->teacher->last_name;
                $data['role'] = 'teacher';
                $data['profile_image'] = $user->teacher->profile_image ? asset('storage/'.$user->teacher->profile_image) : null;
            } elseif ($user->hasRole('admin')) {
                $data['name'] = 'Administrateur AlloTutor';
                $data['role'] = 'admin';
                $data['profile_image'] = null;
            }

            return $data;
        });

        return response()->json(['contacts' => $formattedUsers]);
    }

    public function newUsers()
    {
        $users = User::role(['student', 'teacher'])
                     ->orderBy('created_at', 'desc')
                     ->take(50)
                     ->get();

        $formattedUsers = $users->map(function ($user) {
            $userData = [];

            if ($user->hasRole('student')) {
                $userData['role'] = 'student';
                $userData['role_id'] = $user->student ? $user->student->id : null;
                $userData['name'] = $user->student ? $user->student->first_name.' '.$user->student->last_name : null;
                $userData['phone_number'] = $user->student ? $user->student->phone_number : null;
                $userData['profile_image'] = $user->student && $user->student->profile_image ? asset('storage/' . $user->student->profile_image) : null;
            } elseif ($user->hasRole('teacher')) {
                $userData['role'] = 'teacher';
                $userData['role_id'] = $user->teacher ? $user->teacher->id : null;
                $userData['name'] = $user->teacher ? $user->teacher->first_name.' '.$user->teacher->last_name : null;
                $userData['phone_number'] = $user->teacher ? $user->teacher->phone_number : null;
                $userData['profile_image'] = $user->teacher && $user->teacher->profile_image ? asset('storage/' . $user->teacher->profile_image) : null;
            }

            return array_merge([
                'id' => $user->id,
                'email' => $user->email,
                'created_at' => $user->created_at,
            ], $userData);
        });

        return response()->json(['users' => $formattedUsers]);
    }

    public function getStudentsTeachersList()
    {
        $users = User::role(['student', 'teacher'])
                     ->orderBy('created_at', 'desc')
                     ->take(50)
                     ->get();

        $formattedUsers = $users->map(function ($user) {
            $userData = [];

            if ($user->hasRole('student')) {
                $userData['role'] = 'student';
                $userData['role_id'] = $user->student ? $user->student->id : null;
                $userData['name'] = $user->student ? $user->student->first_name.' '.$user->student->last_name : null;
                $userData['phone_number'] = $user->student ? $user->student->phone_number : null;
                $userData['profile_image'] = $user->student && $user->student->profile_image ? asset('storage/' . $user->student->profile_image) : null;
            } elseif ($user->hasRole('teacher')) {
                $userData['role'] = 'teacher';
                $userData['role_id'] = $user->teacher ? $user->teacher->id : null;
                $userData['name'] = $user->teacher ? $user->teacher->first_name.' '.$user->teacher->last_name : null;
                $userData['phone_number'] = $user->teacher ? $user->teacher->phone_number : null;
                $userData['profile_image'] = $user->teacher && $user->teacher->profile_image ? asset('storage/' . $user->teacher->profile_image) : null;
            }

            return array_merge([
                'id' => $user->id,
                'email' => $user->email,
                'status' => $user->status,
                'created_at' => $user->created_at,
            ], $userData);
        });

        return response()->json(['users' => $formattedUsers]);
    }

    public function show(User $user)
    {
        $user->load('notificationPref');

        return response()->json(['user' => $user]);
    }

    public function getSecret(User $user)
    {
        $stripe = new \Stripe\StripeClient(env('STRIPE_SECRET'));

        if (!$user->stripe_id) {
            $customer = $stripe->customers->create([
                'email' => $user->email,
                'name' => $user->first_name.' '.$user->last_name,
            ]);

            $user->stripe_id = $customer->id;
            $user->save();
        }

        $intent = $stripe->setupIntents->create([
            'customer' => $user->stripe_id,
            'automatic_payment_methods' => ['enabled' => true],
        ]);

        return response()->json(['client_secret' => $intent->client_secret]);
    }

    public function register(Request $request)
    {
        $validatedData = $request->validate([
            'email' => 'required|email|unique:users',
            'role' => 'required|exists:roles,name',
            'password' => 'required|string|confirmed',
            'timezone' => 'nullable|string',
        ]);

        $validatedData['timezone'] = $validatedData['timezone'] ?? 'UTC';
        $role = Role::where('name', $validatedData['role'])->first();
        unset($validatedData['role']);
        $validatedData['password'] = Hash::make($validatedData['password']);

        DB::beginTransaction();

        try {
            // Create the user
            $user = User::create($validatedData);
            $user->assignRole($role);

            DB::commit();

            return $user;
        } catch (\Exception $e) {
            // Rollback transaction and handle error
            DB::rollBack();

            return response()->json(['error' => 'Registration failed: '.$e->getMessage()], 500);
        }
    }

    public function login(Request $request)
    {
        $validatedData = $request->validate([
            'email' => 'required|email',
            'password' => 'required',
        ]);

        // Check for soft deleted user first
        $user = User::withTrashed()
        ->where('email', $validatedData['email'])
        ->first();

        if (!$user) {
            return response()->json(['message' => 'Account Not found'], 404);
        }

        // If account is deleted
        if ($user->trashed()) {
            return response()->json([
                'message' => 'This account has been deleted',
                'isDeleted' => true,
            ], 200);
        }

        // For active accounts, verify password
        if (!Hash::check($validatedData['password'], $user->password)) {
            return response()->json(['message' => 'Wrong password'], 401);
        }

        $roles = $user->getRoleNames()->toArray();
        if (in_array('admin', $roles)) {
            $role = 'admin';
        } elseif (in_array('teacher', $roles)) {
            $role = 'teacher';
            $roleId = $user->teacher->id;
            $firstName = $user->teacher->first_name;
            $lastName = $user->teacher->last_name;
            $status = $user->teacher->status;
            $profile_image = $user->teacher->profile_image ? asset('storage/'.$user->teacher->profile_image) : null;
            $completionRate = (new TeacherController())->calculateProfileCompletionRate($user->teacher);
        } elseif (in_array('student', $roles)) {
            $role = 'student';
            $roleId = $user->student->id;
            $firstName = $user->student->first_name;
            $lastName = $user->student->last_name;
            $status = $user->student->status;
            $profile_image = $user->student->profile_image ? asset('storage/'.$user->student->profile_image) : null;
            $credits = (new StudentController())->getCredits($user->student);
        } else {
            return response()->json(['message' => 'User has no role'], 401);
        }

        // Modify user data before returning
        $userData = [
            'id' => $user->id,
            'role_id' => $roleId ?? null,
            'timezone' => $user->timezone,
            'email' => $user->email,
            'first_name' => $firstName ?? null,
            'last_name' => $lastName ?? null,
            'status' => $status ?? null,
            'profile_image' => $profile_image ?? null,
        ];

        $token = $user->createToken('auth-token')->plainTextToken;
        if (!$user->timezone) {
            $user->update(['timezone' => $request->header('timezone')]);
        }
        if (isset($completionRate)) {
            $userData['completion_rate'] = $completionRate;
        }
        if (isset($credits)) {
            $userData['credits'] = $credits;
        }

        return response()->json(
            [
                'user' => $userData,
                'token' => $token,
                'role' => $role,
            ],
            200
        );
    }

    public function requestRestore(Request $request)
    {
        $validatedData = $request->validate([
            'email' => 'required|email'
        ]);

        $user = User::onlyTrashed()
            ->where('email', $validatedData['email'])
            ->first();

        if (!$user) {
            return response()->json(['message' => 'Account not found'], 404);
        }

        // Generate restore token
        $restoreToken = Str::random(64);
        Cache::put('account_restore_' . $restoreToken, $user->id, now()->addHours(24));

        // Send notification
        $user->notify(new AccountRestoreNotification($restoreToken));

        return response()->json([
            'message' => 'Restoration link sent to your email',
            'message_code' => 'RESTORE_EMAIL_SENT'
        ]);
    }

    public function restore(Request $request)
    {
        try {
            // Validate request data
            $validatedData = $request->validate([
                'restore_token' => 'nullable|string',
                'user_id' => 'nullable|integer',
            ]);
    
            $restoreToken = $validatedData['restore_token'] ?? null; // Safely handle missing restore_token
            $userId = null;
            $isAdmin = $request->user() ? $request->user()->hasRole('admin') : null;
    
            if ($restoreToken) {
                // Try to retrieve userId from cache using restore token
                $userId = Cache::get('account_restore_' . $restoreToken);
            }
    
            // Check if the token is invalid and the user is not an admin
            if (!$userId && !$isAdmin) {
                return response()->json([
                    'message' => 'Le lien de restauration est invalide ou a expiré',
                    'message_code' => 'ACCOUNT_RESTORE_INVALID_TOKEN',
                ], 400);
            }
    
            // Admin can provide user_id directly
            if ($isAdmin && $validatedData['user_id']) {
                $userId = $validatedData['user_id'];
            }
    
            // Find the soft-deleted user by ID
            $user = User::withTrashed()->findOrFail($userId);
    
            // Check if the user is already active
            if (!$user->trashed()) {
                return response()->json([
                    'message' => 'Le compte est déjà actif ou n\'a pas été supprimé',
                    'message_code' => 'ACCOUNT_ALREADY_ACTIVE',
                    'user' => $user,
                ], 400);
            }
    
            DB::beginTransaction();
    
            // Restore user and update status
            $user->restore();
            $user->status = 'active';
            $user->save();
    
            // Update status for teacher or student role
            if ($user->hasRole('teacher')) {
                $user->teacher->update(['status' => 'active']);
            } elseif ($user->hasRole('student')) {
                $user->student->update(['status' => 'active']);
            }
    
            // Create default notification preferences
            $user->notificationPref()->create([
                'new_lessons' => true,
                'lesson_updates' => true,
                'new_messages' => true,
                'events_promotions' => true,
                'app_updates' => true,
                'lesson_reminders' => true,
                'reminder_time' => '60',
                'email_notifications' => true,
                'in_app_notifications' => true,
            ]);
    
            // Remove the restore token from cache
            if ($restoreToken) {
                Cache::forget('account_restore_' . $restoreToken);
            }
    
            // Notify the user (optional)
            if ($isAdmin) {
                $user->notify(new AccountRestoreNotification(null, $isAdmin));
            }
    
            DB::commit();
    
            return response()->json([
                'message' => 'Compte restauré avec succès. Vous pouvez maintenant vous connecter.',
                'message_code' => 'ACCOUNT_RESTORE_SUCCESS',
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
    
            Log::error('Account restoration failed', [
                'restore_token' => $request->input('restore_token'),
                'error' => $e->getMessage(),
            ]);
    
            return response()->json([
                'message' => 'Échec de la restauration du compte',
                'message_code' => 'ACCOUNT_RESTORE_ERROR',
            ], 500);
        }
    }
    

    public function logout(Request $request)
    {
        $request->user()->currentAccessToken()->delete();

        return response()->json(['message' => 'User logout successfully']);
    }

    public function logoutFromAllDevices(Request $request)
    {
        // Get the currently authenticated user
        $user = $request->user();

        // Delete all access tokens for this user
        $user->tokens()->delete(); // This removes all tokens for the user

        // Return a success message
        return response()->json(['message' => 'Successfully logged out from all devices']);
    }

    public function updatePassword(Request $request, User $user)
    {
        $validatedData = $request->validate([
            'old_password' => 'required|string',
            'password' => 'required|string|confirmed',
        ]);

        // Check if the old password matches the user's actual password
        if (!Hash::check($validatedData['old_password'], $user->password)) {
            return response()->json([
                'message' => 'Old password is incorrect',
                'message_code' => 'WRONG_PASSWORD'
            ], 400); // 422 Unprocessable Entity
        }

        unset($validatedData['old_password']);
        $validatedData['password'] = Hash::make($validatedData['password']);

        $user->update($validatedData);
        $this->logoutFromAllDevices($request);

        return response()->json([
            'user' => $user,
            'message' => 'Password updated successfully!',
        ]);
    }

    public function updateTimezone(Request $request, User $user)
    {
        $validatedData = $request->validate([
            'timezone' => 'required|string',
        ]);

        $user->update($validatedData);

        return response()->json([
            'message' => 'User timezone updated successfully!',
            'message_code' => 'UPDATE_TIMEZONE_SUCCESS',
        ]);
    }

    public function getTimezone(User $user)
    {
        return response()->json([
            'timezone' => $user->timezone,
        ]);
    }

    public function destroy(Request $request, $id)
    {
        $validatedData = $request->validate([
            'token' => 'required|string',
            'permanentDelete' => 'required|boolean',
        ]);

        // Verify token from previous step
        $cachedToken = Cache::get('identity_verification_' . $request->user()->id);
        if (!$cachedToken || $cachedToken !== $validatedData['token']) {
            return response()->json([
                'message' => 'Session invalide or expired.',
                'message_code' => 'INVALID_IDENTITY_TOKEN'
            ], 400);
        }

        try {
            DB::beginTransaction();
            $user = User::withTrashed()->findOrFail($id);
            $teacher = null;
            $student = null;
            $requestedBy = $request->user()->hasRole('admin') ? 'admin' : 'self';

            if ($user->hasRole('teacher')) {
                $teacher = $user->teacher;

                if ($teacher->balance > 0) {
                    return response()->json([
                        'message' => 'Cannot delete account while balance is not zero',
                        'message_code' => 'ACCOUNT_DELETE_BALANCE_ERROR'
                    ], 422);
                }
            } else if ($user->hasRole('student')) {
                $student = $user->student;

                if ((new StudentController())->getCredits($student) > 0) {
                    return response()->json([
                        'message' => 'Cannot delete account while credits are not zero',
                        'message_code' => 'ACCOUNT_DELETE_CREDITS_ERROR'
                    ], 422);
                }
            }

            // Handle permanent deletion if requested and the authenticated user is an admin
            if ($validatedData['permanentDelete'] && $requestedBy === 'admin') {
                $user->tokens()->delete();
                $user->forceDelete(); 
    
                // Send notification to the user
                $user->notify(new AccountDeletedNotification($requestedBy,false));

                DB::commit();

                return response()->json([
                    'message' => 'Account permanently deleted successfully',
                    'message_code' => 'ACCOUNT_DELETE_PERMANENTLY_SUCCESS',
                ]);
            }

            // Cancel lessons and notify other users
            $lessons = Lesson::where(function ($query) use ($student, $teacher) {
                if ($student) {
                    $query->orWhere('student_id', $student->id);
                }
                if ($teacher) {
                    $query->orWhere('teacher_id', $teacher->id);
                }
            })->get();

            foreach ($lessons as $lesson) {
                $passedStatus = $lesson->status;
                $lesson->status = 'canceled';
                $lesson->save();

                $student = $lesson->student;
                $teacher = $lesson->teacher;

                if($passedStatus == 'scheduled') {
                    if ($student && $student->user_id !== $user->id) {
                        $receiver = User::findOrFail($student->user_id);
                        $receiver->notify(new LessonNotification($lesson, 'canceled', 'student', $receiver->id));
                    }

                    if ($teacher && $teacher->user_id !== $user->id) {
                        $receiver = User::findOrFail($teacher->user_id);
                        $receiver->notify(new LessonNotification($lesson, 'canceled', 'teacher', $receiver->id));
                    }
                } 
                if($passedStatus == 'pending' && $student && $student->user_id !== $user->id) {
                    $createdBy = User::find($lesson->created_by);
                    if ($createdBy->hasRole('admin')) {
                        $createdBy->notify(new LessonNotification($lesson, 'rejected', 'admin', $createdBy->id));
                    }
                    if($lesson->type == 'normal'){
                        $receiver = User::findOrFail($student->user_id);
                        $receiver->notify(new LessonNotification($lesson, 'rejected', 'student', $receiver->id));
                    }
                } 
                if ($lesson->lesson_type == 'trial') {
                    if ($lesson->teacher_id) {
                        $lesson->teacher_id = null;
                        $lesson->status = 'pending_schedule';
                        $lesson->save();
                    }
                } 
            }

            // Handle teacher-specific deletion
            if ($teacher) {
                $modifyFields = [
                    'biography' => null,
                    'education' => null,
                    'experience' => null,
                    'gender' => null,
                    'address' => null,
                    'profile_image' => null,
                    'country' => null,
                    'language' => null,
                    'age' => null,
                    'postal_code' => null,
                    'city' => null,
                    'actual_job' => null,
                    'interests' => null,
                    'balance' => 0,
                    'status' => 'deleted',
                ];

                $teacher->update($modifyFields);
                $teacher->courses()->detach();
                $teacher->availabilities()->delete();
            }
            // Handle student-specific deletion
            if ($student) {

                $modifyFields = [
                    'learning_objective' => null,
                    'gender' => null,
                    'address' => null,
                    'profile_image' => null,
                    'interests' => null,
                    'age' => null,
                    'postal_code' => null,
                    'city' => null,
                    'country' => null,
                    'status' => 'deleted',
                ];

                $student->update($modifyFields);
                $student->courses()->detach();
            }

            // Detach payment methods from Stripe
            $stripe = new \Stripe\StripeClient(env('STRIPE_SECRET'));
            foreach ($user->paymentMethods as $paymentMethod) {
                $stripe->paymentMethods->detach($paymentMethod->stripe_payment_method_id);
            }

            // Delete user-specific relationships
            $user->reports()->delete();
            $user->notificationPref()->delete();
            $user->accountPauseHistories()->delete();
            $user->paymentMethods()->delete();
            $user->notifications()->delete();

            // Delete tokens
            $user->tokens()->delete();

            // Soft delete the user
            $user->timezone = null;
            $user->status = 'deleted';
            $user->save();
            $user->delete();

            // Send notification to the user
            $user->notify(new AccountDeletedNotification($requestedBy,true));

            DB::commit();

            return response()->json([
                'message' => 'User deleted successfully',
                'message_code' => 'ACCOUNT_DELETE_SUCCESS',
                'lessons' => $lessons,
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            Log::error('Account deletion failed', [
                'user_id' => $user->id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'message' => 'Failed to delete account',
                'message_code' => 'ACCOUNT_DELETE_ERROR',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function PermanentDelete(User $user)
    {
        $user->tokens()->delete();
        $user->forceDelete();

        return response()->json([
            'message' => 'Account deleted permenantly successfully',
            'message_code' => 'ACCOUNT_DELETE_PERMANENTLY_SUCCESS',
        ]);
    }

    public function sendPasswordResetLink(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
        ]);

        // Check for soft deleted user first
        $user = User::withTrashed()
        ->where('email', $request->email)
        ->first();

        if (!$user) {
            return response()->json([
                'message' => 'Email address not found.',
                'message_code' => 'EMAIL_NOT_FOUND',
            ], 404);
        }

        // If account is deleted
        if ($user->trashed()) {
            return response()->json([
                'message' => 'This account has been deleted',
                'isDeleted' => true,
            ], 200);
        }

        $token = Password::createToken($user, now()->addMinutes(config('auth.passwords.users.expire')));

        $user->notify(new ResetPasswordNotification($token));

        return response()->json([
            'message' => 'Password reset link sent to your email address.',
        ]);
        
    }

    public function verifyIdentity(Request $request, User $user)
    {
        $request->validate([
            'password' => 'required|string',
        ]);
        
        if (!Hash::check($request->password, $user->password)) {
            return response()->json([
                'message' => 'Le mot de passe est incorrect.',
                'message_code' => 'WRONG_PASSWORD'
            ], 400);
        }

        // Generate a temporary token for email update
        $token = Str::random(60);
        Cache::put('identity_verification_' . $user->id, $token, now()->addMinutes(30));

        return response()->json([
            'message' => 'Password verified successfully',
            'token' => $token
        ]);
    }

    public function initiateEmailUpdate(Request $request, User $user)
    {
        $request->validate([
            'newEmail' => 'required|email',
            'token' => 'required|string'
        ]);
        
        // Verify token from previous step
        $cachedToken = Cache::get('identity_verification_' . $user->id);
        if (!$cachedToken || $cachedToken !== $request->token) {
            return response()->json([
                'message' => 'Session invalide or expired',
                'message_code' => 'INVALID_IDENTITY_TOKEN'
            ], 400);
        }

        // Check if the new email is unique
        if (User::where('email', $request->newEmail)->exists()) {
            return response()->json([
                'message' => 'Email already in use',
                'message_code' => 'EMAIL_ALREADY_IN_USE'
            ], 400);
        }

        // Generate confirmation code
        $confirmationCode = sprintf('%06d', mt_rand(0, 999999));
        Cache::put('email_confirmation_' . $user->id, [
            'code' => $confirmationCode,
            'email' => $request->newEmail
        ], now()->addMinutes(15));

        // Send confirmation email
        Mail::to($request->newEmail)->send(new EmailUpdateConfirmation($confirmationCode));

        return response()->json([
            'message' => 'Code de confirmation envoyé au nouvel email.',
            'message_code' => 'EMAIL_UPDATE_CONFIRMATION_CODE_SEND'
        ]);
    }

    public function confirmEmailUpdate(Request $request, User $user)
    {
        $request->validate([
            'confirmationCode' => 'required|string'
        ]);
        
        $cached = Cache::get('email_confirmation_' . $user->id);
        if (!$cached) {
            return response()->json([
                'message' => 'Code de confirmation expiré ou invalide.',
                'message_code' => 'INVALID_CONFIRMATION_CODE'
            ], 400);
        } elseif ($cached['code'] !== $request->confirmationCode) {
            return response()->json([
                'message' => 'Code de confirmation incorrect.',
                'message_code' => 'WRONG_CONFIRMATION_CODE'
            ], 400);
        }

        try {
            DB::beginTransaction();

            // Update user email
            $user->email = $cached['email'];
            $user->save();

            // Update role-specific email
            if ($user->hasRole('student')) {
                Student::where('user_id', $user->id)->update(['email' => $cached['email']]);
            } elseif ($user->hasRole('teacher')) {
                Teacher::where('user_id', $user->id)->update(['email' => $cached['email']]);
            }

            // Logout from all devices
            $user->tokens()->delete();
            
            DB::commit();
            Cache::forget('email_confirmation_' . $user->id);
            Cache::forget('identity_verification_' . $user->id);

            return response()->json([
                'message' => 'Email mis à jour avec succès.',
                'message_code' => 'EMAIL_UPDATE_SUCCESS'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => 'Erreur lors de la mise à jour de l\'email.',
            ], 500);
        }
    }

    public function authenticate(Request $request)
    {
        try {
            $socketId = $request->input('socket_id');
            $channelName = $request->input('channel_name');

            if (!$socketId || !$channelName) {
                return response()->json(['error' => 'Invalid request parameters'], 400);
            }

            $pusher = new Pusher(
                env('PUSHER_APP_KEY'),
                env('PUSHER_APP_SECRET'),
                env('PUSHER_APP_ID'),
                [
                    'cluster' => env('PUSHER_APP_CLUSTER'),
                    'useTLS' => true,
                ]
            );

            $user = $request->user();
            if (!$user) {
                return response()->json(['error' => 'User not authenticated'], 403);
            }

            $presenceData = [
                'user_id' => $user->id,
                'user_info' => [
                    'email' => $user->email,
                ],
            ];

            $auth = $pusher->presence_auth($channelName, $socketId, $user->id, $presenceData);

            return response()->json(json_decode($auth, true));
        } catch (\Exception $e) {
            return response()->json(['error' => 'Authentication failed'], 500);
        }
    }

    public function resetPassword(Request $request)
    {
        $request->validate([
            'token' => 'required',
            'email' => 'required|email',
            'password' => 'required|confirmed|min:8',
        ]);

        $status = Password::broker()->reset(
            $request->only('email', 'password', 'token'),
            function ($user, $password) {
                $user->forceFill([
                    'password' => Hash::make($password),
                ])->save();

                $user->tokens()->delete(); // Delete the used reset token
            }
        );

        if ($status === Password::PASSWORD_RESET) {
            return response()->json([
                'message' => 'Password reset successfully.',
                'message_code' => 'PASSWORD_RESET_SUCCESS',
            ]);
        }

        $errorMessage = __($status); // Get the translated error message

        return response()->json(['message' => $errorMessage], 422);
    }

    public function createStripeCustomerAccount(User $user)
    {
        new \Stripe\StripeClient(env('STRIPE_SECRET'));

        try {
            if ($user->stripe_id) {
                try {
                    // Try to retrieve existing customer
                    $customer = Customer::retrieve($user->stripe_id);
                    return $customer;
                } catch (\Stripe\Exception\InvalidRequestException $e) {
                    // If customer not found or invalid, clear the stripe_id
                    if ($e->getStripeCode() === 'resource_missing' || $e->getHttpStatus() === 404) {
                        $user->stripe_id = null;
                        $user->save();
                    }
                }
            }

            // Create new customer if stripe_id is null or was cleared
            if (!$user->stripe_id) {
                $customer = Customer::create([
                    'email' => $user->email,
                    'name' => $user->first_name.' '.$user->last_name,
                ]);

                $user->stripe_id = $customer->id;
                $user->save();
                
                return $customer;
            }
        } catch (\Exception $e) {
            // Handle other potential errors
            Log::error('Stripe customer creation error: ' . $e->getMessage());
            throw $e;
        }
    }

    public function createStripeConnectAccount(User $user)
    {
        $stripe = new \Stripe\StripeClient(env('STRIPE_SECRET'));

        if (!$user->stripe_id) {
            $account = $stripe->accounts->create([
                'type' => 'custom',
                'country' => 'BE',
                'email' => $user->email,
                'business_type' => 'individual',
                'capabilities' => [
                    'transfers' => ['requested' => true],
                ],
                'tos_acceptance' => [
                    'date' => time(),
                    'ip' => $_SERVER['REMOTE_ADDR'],
                ],
                'individual' => [
                    'first_name' => $user->teacher->first_name,
                    'last_name' => $user->teacher->last_name,
                    'email' => $user->email,
                    'dob' => [
                        'day' => 01,
                        'month' => 01,
                        'year' => 2000,
                    ],
                    'address' => [
                        'line1' => $user->teacher->address,
                        'city' => $user->teacher->city,
                        'postal_code' => $user->teacher->postal_code,
                    ],
                ],
                'business_profile' => [
                    'product_description' => 'course', // Explicitly set an empty URL
                    'mcc' => '8299',
                ],
                'settings' => [
                    'payouts' => [
                        'schedule' => [
                            'interval' => 'manual',
                        ],
                    ],
                ],
            ]);

            $user->stripe_id = $account->id;
            $user->save();
        } else {
            return response()->json(['error' => 'Connect account already exist'], 409);
        }

        return response()->json([
            'message' => 'Connect account created successfully!',
        ]);
    }

    public function addBankDetail(Request $request, User $user)
    {
        $stripe = new \Stripe\StripeClient(env('STRIPE_SECRET'));

        $validatedData = $request->validate([
            'token' => 'required',
        ]);

        try {
            $stripe->accounts->createExternalAccount(
                $user->stripe_id,
                ['external_account' => $validatedData['token']]
            );

            return response()->json([
                'message' => 'Bank detail added successfully!',
            ]);
        } catch (\Stripe\Exception\ApiErrorException $e) {
            return response()->json(['error' => 'Failed to add bank detail: '.$e->getMessage()], 500);
        }
    }

    public function attachCardToCustomer(Request $request, User $user)
    {
        Stripe::setApiKey(env('STRIPE_SECRET'));

        $validatedData = $request->validate([
            'payment_method_id' => 'required|string',
            'default_card' => 'required|boolean',
        ]);

        $customer = $this->createStripeCustomer($user);
        $paymentMethod = PaymentMethod::retrieve($validatedData['payment_method_id']);
        $paymentMethod->attach(['customer' => $customer->id]);

        if ($validatedData['default_card']) {
            $stripe = new \Stripe\StripeClient(env('STRIPE_SECRET'));

            $stripe->customers->update(
                $customer->id,
                ['invoice_settings' => ['default_payment_method' => $paymentMethod->id]]
            );
        }

        return response()->json([
            'message' => 'Card attached successfully!',
            'customer_id' => $customer->id,
            'payment_method' => $paymentMethod,
        ]);
    }

    public function getBankDetail(User $user)
    {
        $stripe = new \Stripe\StripeClient(env('STRIPE_SECRET'));

        try {
            $bankDetails = $stripe->accounts->allExternalAccounts(
                $user->stripe_id,
                ['object' => 'bank_account']
            );

            return response()->json(['bank_detail' => $bankDetails->data], 200);
        } catch (\Stripe\Exception\ApiErrorException $e) {
            return response()->json(['error' => 'An error occurred while fetching bank detail : '.$e->getMessage()], 500);
        }
    }

    public function getPaymentMethods(User $user)
    {
        Stripe::setApiKey(env('STRIPE_SECRET'));
        $customer = $this->createStripeCustomerAccount($user);
        $defaultPaymentMethodId = $customer->invoice_settings->default_payment_method;

        try {
            $paymentMethods = $user->paymentMethods()->where('status', 'active')->get();
            $paymentMethods = $paymentMethods->map(function ($paymentMethod) use ($defaultPaymentMethodId) {
                $paymentMethod->is_default = $paymentMethod->stripe_payment_method_id === $defaultPaymentMethodId;

                return $paymentMethod;
            });

            return response()->json(['payment_methods' => $paymentMethods], 200);
        } catch (\Stripe\Exception\CardException $e) {
            return response()->json(['error' => $e->getMessage()], 400);
        } catch (\Exception $e) {
            return response()->json(['error' => 'An error occurred while fetching payment methods : '.$e->getMessage()], 400);
        }
    }

    public function updateDefaultPaymentMethod(Request $request, User $user)
    {
        $validatedData = $request->validate([
            'stripe_payment_method_id' =>  'nullable|string',
        ]);

        $stripe = new \Stripe\StripeClient(env('STRIPE_SECRET'));

         // If payment_method_id is null, remove default payment method
        if ($validatedData['stripe_payment_method_id'] === null) {
            $stripe->customers->update(
                $user->stripe_id,
                ['invoice_settings' => ['default_payment_method' => null]]
            );

            return response()->json([
                'message' => 'Default payment method removed successfully!',
                'message_code' => 'DEFAULT_PAYMENT_METHOD_REMOVED_SUCCESS',
            ]);
        }

         // Set default payment method
        $stripe->customers->update(
            $user->stripe_id,
            ['invoice_settings' => ['default_payment_method' => $validatedData['stripe_payment_method_id']]]
        );

        return response()->json([
            'message' => 'Payment method set as default successfully!',
            'message_code' => 'DEFAULT_PAYMENT_METHOD_SET_SUCCESS',
        ]);
    }

    public function getDiscussionList(User $user)
    {
        $messages = Message::where('sender_id', $user->id)
            ->orWhere('receiver_id', $user->id)
            ->with(['sender:id,email', 'receiver:id,email'])
            ->latest('created_at') // This ensures messages are ordered by creation date
            ->get();

        // Group by conversation partner and maintain the latest message order
        $discussionList = $messages
            ->groupBy(function ($message) use ($user) {
                return $message->sender_id === $user->id ? $message->receiver_id : $message->sender_id;
            })
            ->map(function ($messages) use ($user) {
                $otherUser = $messages->first()->sender_id === $user->id
                    ? User::withTrashed()->find($messages->first()->receiver_id)
                    : User::withTrashed()->find($messages->first()->sender_id);

                $unreadMessageCount = $messages->where('receiver_id', $user->id)
                    ->where('read', false)
                    ->whereNull('read_at')
                    ->count();

                $lastMessageTimestamp = $messages->first()->created_at;

                // Base discussion data
                $discussionData = [
                    'id' => $otherUser->id,
                    'unreadMessageCount' => $unreadMessageCount,
                    'last_message_at' => $lastMessageTimestamp,
                    'last_message' => $messages->first()->message,
                    'last_message_you' => $messages->first()->sender_id != $otherUser->id ,
                ];

                // Add user-specific data based on role
                if ($otherUser->hasRole('admin')) {
                    return array_merge($discussionData, [
                        'name' => 'AlloTutor',
                    ]);
                } elseif ($otherUser->hasRole('teacher')) {
                    return array_merge($discussionData, [
                        'name' => $otherUser->teacher->first_name.' '.$otherUser->teacher->last_name,
                        'status' => $otherUser->status,
                        'profile_image' => $otherUser->teacher->profile_image 
                            ? asset('storage/'.$otherUser->teacher->profile_image) 
                            : null,
                    ]);
                } elseif ($otherUser->hasRole('student')) {
                    return array_merge($discussionData, [
                        'name' => $otherUser->student->first_name.' '.$otherUser->student->last_name,
                        'status' => $otherUser->status,
                        'profile_image' => $otherUser->student->profile_image 
                            ? asset('storage/'.$otherUser->student->profile_image) 
                            : null,
                    ]);
                } else {
                    return array_merge($discussionData, [
                        'name' => 'Inconnue',
                    ]);
                }
            })
            ->sortByDesc('last_message_at') // Sort discussions by the last message timestamp
            ->values();

        return response()->json([
            'discussions' => $discussionList,
            'total_unread' => $messages->where('receiver_id', $user->id)
                ->where('read', false)
                ->whereNull('read_at')
                ->count()
        ]);
    }

    public function getDiscussion(User $user, $otherUserId, Request $request)
    {
        $userId = $user->id;
        $otherUser = User::withTrashed()->find($otherUserId);
        $limit = $request->input('limit', 20); // Nombre de messages à charger par défaut
        $page = $request->input('page', 1);
        
        $messages = Message::where(function ($query) use ($userId, $otherUserId) {
            $query->where('sender_id', $userId)
                ->where('receiver_id', $otherUserId);
        })->orWhere(function ($query) use ($userId, $otherUserId) {
            $query->where('sender_id', $otherUserId)
                ->where('receiver_id', $userId);
        })
        ->orderBy('created_at', 'desc') // Changé de 'asc' à 'desc' pour obtenir les plus récents d'abord
        ->paginate($limit, ['*'], 'page', $page);
        
        // On renverse l'ordre pour l'affichage correct
        $messagesCollection = $messages->getCollection()->reverse()->values();
        $messages->setCollection($messagesCollection);
        
        return response()->json([
            'messages' => $messages->items(),
            'other_user' => $otherUser,
            'has_more' => $messages->hasMorePages(),
            'total' => $messages->total(),
            'limit' => $limit,
            'page' => $page,
        ]);
    }

    public function deleteDiscussion(User $user, User $otherUser)
    {
        try {
            $userId = $user->id;
            $otherUserId = $otherUser->id;

            $deletedCount = Message::where(function ($query) use ($userId, $otherUserId) {
                $query->where('sender_id', $userId)
                    ->where('receiver_id', $otherUserId);
            })->orWhere(function ($query) use ($userId, $otherUserId) {
                $query->where('sender_id', $otherUserId)
                ->where('receiver_id', $userId);
            })->delete();

            return response()->json([
                'message_code' => 'DISCUSSION_DELETE_SUCCESS',
                'message' => 'Discussion deleted successfully',
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Failed to delete discussion',
                'error' => $e->getMessage()
            ], 400);
        }
    }

    public function markDiscussionMessagesAsRead(User $user, User $otherUser)
    {
        $messages = Message::where('receiver_id', $user->id)
            ->where('sender_id', $otherUser->id)
            ->where('read', false)
            ->whereNull('read_at')
            ->get();

        $messages->each(function ($message) {
            $message->update([
                'read_at' => now(),
                'read' => true,
            ]);
        });

        event(new AllMessagesRead($user, $otherUser, $messages));

        return response()->json(['message' => 'Messages marked as read successfully.']);
    }

    public function updateNotificationPreferences(Request $request, User $user)
    {
        try {
            DB::beginTransaction();
    
            $validatedData = $request->validate([
                'new_lessons' => 'boolean',
                'lesson_updates' => 'boolean',
                'new_messages' => 'boolean',
                'events_promotions' => 'boolean',
                'app_updates' => 'boolean',
                'lesson_reminders' => 'boolean',
                'reminder_time' => 'in:15,60,180,360',
                'email_notifications' => 'boolean',
                'in_app_notifications' => 'boolean',
            ]);
    
            $preferences = $user->notificationPref;
            
            if (!$preferences) {
                throw new \Exception('Notification preferences not found for this user.');
            }
    
            $preferences->update($validatedData);
    
            DB::commit();
    
            return response()->json([
                'message' => 'Preferences updated successfully!',
                'message_code' => 'UPDATE_NOTIFICATION_PREF_SUCCESS',
            ]);
    
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error updating notification preferences: ' . $e->getMessage());
    
            return response()->json([
                'message' => 'An error occurred while updating notification preferences',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    public function resetNotificationPreferences(User $user)
    {
        $preferences = $user->notificationPreferences;
        $preferences->update([
            'new_lessons' => true,
            'lesson_updates' => true,
            'student_messages' => true,
            'events_promotions' => true,
            'app_updates' => true,
            'lesson_reminders' => true,
            'reminder_time' => '60',
            'email_notifications' => false,
            'in_app_notifications' => true,
        ]);

        return response()->json([
            'message' => 'Preferences reset to default!',
            'message_code' => 'NOTIFICATION_PREF_RESET_DEFAULT_SUCCESS',
        ]);
    }
}
