<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\Lesson;
use App\Models\Payment;
use App\Models\Teacher;
use App\Models\TeacherPayment;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Http\Request;

class TeacherPaymentController extends Controller
{
    private const EARNINGS_COMPLETED = 4000;
    private const EARNINGS_STUDENT_ABSENT = 1000;
    private const PENALTY_TEACHER_ABSENT = 4500;
    private const EARNINGS_TRIAL_COMPLETED = 2000;

    public function getTeachetRevenueData(Teacher $teacher)
    {
        $currentMonth = Carbon::now()->startOfMonth();

        return response()->json([
            'currentBalance' => $teacher->balance,
            'totalRevenue' => $this->getTotalRevenue($teacher),
            'monthlyStats' => $this->getMonthlyStats($teacher, $currentMonth),
            'earningsDetails' => $this->getEarningsDetails(),
            'passedLessonsEarning' => $this->getLessonsEarning($teacher),
            'payments' => $teacher->payments()->with('teacher:id,first_name,last_name')->get(),
        ]);
    }

    private function getTotalRevenue(Teacher $teacher)
    {
        $completedNormalLessons = Lesson::where('teacher_id', $teacher->id)
            ->where('status', 'completed')
            ->where('lesson_type', 'normal')
            ->count();

        $completedTrialLessons = Lesson::where('teacher_id', $teacher->id)
            ->where('status', 'completed')
            ->where('lesson_type', 'trial')
            ->count();

        $studentAbsentLessons = Lesson::where('teacher_id', $teacher->id)
            ->where('status', 'expired')
            ->where('teacher_present', true)
            ->where('student_present', false)
            ->count();

        $teacherAbsentLessons = Lesson::where('teacher_id', $teacher->id)
            ->where('status', 'expired')
            ->where('teacher_present', false)
            ->where('student_present', true)
            ->count();

        return ($completedNormalLessons * self::EARNINGS_COMPLETED) +
               ($completedTrialLessons * self::EARNINGS_TRIAL_COMPLETED) +
               ($studentAbsentLessons * self::EARNINGS_STUDENT_ABSENT) -
               ($teacherAbsentLessons * self::PENALTY_TEACHER_ABSENT);
    }

    private function getMonthlyStats(Teacher $teacher, $currentMonth)
    {
        $monthlyLessons = Lesson::where('teacher_id', $teacher->id)
            ->whereMonth('date', $currentMonth->month)
            ->whereYear('date', $currentMonth->year)->get();

        $passedLessons = $monthlyLessons->whereIn('status', ['expired', 'completed'])->count();
        $completedNormalLessons = $monthlyLessons->where('status', 'completed')
                                                ->where('lesson_type', 'normal')
                                                ->count();

        $completedTrialLessons = $monthlyLessons->where('status', 'completed')
                                                ->where('lesson_type', 'trial')
                                                ->count();
        $studentAbsentLessons = $monthlyLessons->where('status', 'expired')
                                            ->where('teacher_present', true)  // Compare with boolean true
                                            ->where('student_present', false)
                                            ->count();
        $teacherAbsentLessons = $monthlyLessons->where('status', 'expired')
                                            ->where('teacher_present', false)
                                            ->where('student_present', true)
                                            ->count();

        $revenue = ($completedNormalLessons * self::EARNINGS_COMPLETED) +
                   ($completedTrialLessons * self::EARNINGS_TRIAL_COMPLETED) +
                   ($studentAbsentLessons * self::EARNINGS_STUDENT_ABSENT);
        $losses = $teacherAbsentLessons * self::PENALTY_TEACHER_ABSENT;

        return [
            'revenue' => $revenue,
            'losses' => $losses,
            'lessonCount' => $passedLessons,
        ];
    }

    public function getEarningsDetails()
    {
        return [
            'completedLessons' => self::EARNINGS_COMPLETED,
            'studentAbsentLessons' => self::EARNINGS_STUDENT_ABSENT,
            'teacherAbsentLessons' => -self::PENALTY_TEACHER_ABSENT,
            'trialLessonCompleted' => self::EARNINGS_TRIAL_COMPLETED,
        ];
    }

    private function getLessonsEarning(Teacher $teacher)
    {

        return Lesson::where('teacher_id', $teacher->id)
            ->whereIn('status', ['completed', 'expired'])
            ->with(['course', 'student'])
            ->orderBy('date', 'desc')
            ->get()
            ->map(function ($lesson) {
                $amount = match (true) {
                    $lesson->status === 'completed' && $lesson->lesson_type === 'normal' => self::EARNINGS_COMPLETED,
                    $lesson->status === 'completed' && $lesson->lesson_type === 'trial' => self::EARNINGS_TRIAL_COMPLETED,
                    $lesson->status === 'expired' && !$lesson->teacher_present && $lesson->student_present => -self::PENALTY_TEACHER_ABSENT,
                    $lesson->status === 'expired' && !$lesson->student_present && $lesson->teacher_present => self::EARNINGS_STUDENT_ABSENT,
                    default => 0,
                };

                $transactionStatus = $amount !== 0 ? 'processed' : 'pending';

                return [
                    'id' => $lesson->id,
                    'date' => $lesson->date,
                    'time' => $lesson->time,
                    'course_title' => $lesson->course->course_title,
                    'student_name' => $lesson->student->first_name.' '.$lesson->student->last_name,
                    'status' => $lesson->status,
                    'lesson_type' => $lesson->lesson_type,
                    'student_present' => $lesson->student_present,
                    'teacher_present' => $lesson->teacher_present,
                    'amount' => $amount,
                    'transaction_status' => $transactionStatus,
                ];
            });
    }

    public function getMonthlyEarnings(Teacher $teacher)
    {
        $monthlyEarnings = Lesson::where('teacher_id', $teacher->id)
            ->whereIn('status', ['completed', 'expired'])
            ->select(
                DB::raw('TO_CHAR(date, \'YYYY-MM\') as month'), // PostgreSQL date formatting
                DB::raw('COUNT(*) as total_lessons'),
                DB::raw('SUM(CASE 
                WHEN status = \'completed\' AND lesson_type = \'normal\' THEN '.self::EARNINGS_COMPLETED.'
                WHEN status = \'completed\' AND lesson_type = \'trial\' THEN '.self::EARNINGS_TRIAL_COMPLETED.'
                WHEN status = \'expired\' AND student_present = false AND teacher_present = true THEN '.self::EARNINGS_STUDENT_ABSENT.'
                ELSE 0 
            END) as gross_earnings'),
                DB::raw('SUM(CASE 
                WHEN status = \'expired\' AND teacher_present = false AND student_present = true THEN '.-self::PENALTY_TEACHER_ABSENT.'
                ELSE 0 
            END) as deductions')
            )
            ->groupBy('month')
            ->orderBy('month')
            ->get()
            ->map(function ($item) {
                $item->netEarnings = $item->gross_earnings + $item->deductions;

                return $item;
            });

        return response()->json([
            'monthlyEarnings' => $monthlyEarnings,
        ]);
    }

    public function downloadReceipt(Teacher $teacher, $month)
    {
        try {
            $earnings = Lesson::where('teacher_id', $teacher->id)
            ->whereRaw("TO_CHAR(date, 'YYYY-MM') = ?", [$month])
            ->whereIn('status', ['completed', 'expired'])
            ->select(
                DB::raw('COUNT(*) as total_lessons'),
                DB::raw('SUM(CASE 
                    WHEN status = \'completed\' AND lesson_type = \'normal\' THEN '.self::EARNINGS_COMPLETED.'
                    WHEN status = \'completed\' AND lesson_type = \'trial\' THEN '.self::EARNINGS_TRIAL_COMPLETED.'
                    WHEN status = \'expired\' AND student_present = false AND teacher_present = true THEN '.self::EARNINGS_STUDENT_ABSENT.'
                    ELSE 0 
                END) as gross_earnings'),
                    DB::raw('SUM(CASE 
                    WHEN status = \'expired\' AND teacher_present = false AND student_present = true THEN '.-self::PENALTY_TEACHER_ABSENT.'
                    ELSE 0 
                END) as deductions')
            )
            ->first();

            $earnings->net_earnings = $earnings->gross_earnings + $earnings->deductions;

            // Fetch the list of lessons for the month
            $lessons = Lesson::where('teacher_id', $teacher->id)
                ->whereRaw("TO_CHAR(date, 'YYYY-MM') = ?", [$month])
                ->whereIn('status', ['completed', 'expired'])
                ->with(['student', 'course'])
                ->get();

            $monthYear = Carbon::createFromFormat('Y-m', $month)->locale('fr_FR')->isoFormat('MMMM YYYY');

            $pdf = Pdf::loadView('receipts.monthly', [
                'teacher' => $teacher,
                'month' => $monthYear,
                'earnings' => $earnings,
                'lessons' => $lessons
            ]);

            return $pdf->download("receipt_{$month}.pdf");
        } catch (\Exception $e) {
            Log::error('Receipt generation failed: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to generate receipt: '. $e->getMessage()], 500);
        }
    }

    public function downloadTeacherPaymentReceipt(Teacher $teacher, TeacherPayment $payment)
    {
        try {
            // Ensure that the payment status is not pending
            if ($payment->status === 'pending') {
                return response()->json(['error' => 'Cannot generate a receipt for a pending payment.'], 400);
            }

            // Retrieve the teacher and ensure it's the same as the one in the payment
            if ($payment->teacher_id !== $teacher->id) {
                return response()->json(['error' => 'Mismatched teacher and payment information.'], 400);
            }
             // Retrieve the most recent previous payment date before this one
            $previousPayment = TeacherPayment::where('teacher_id', $teacher->id)
            ->where('payment_date', '<', $payment->payment_date)
            ->orderBy('payment_date', 'desc')
            ->first();

             // Check if this is the only payment ever made to the teacher
            $isFirstPayment = !TeacherPayment::where('teacher_id', $teacher->id)->where('id', '!=', $payment->id)->where('status', '=', 'payed')->exists();

            // Determine the start date of the payment period
            if ($previousPayment) {
                $payment_start_date = Carbon::parse($previousPayment->payment_date)->format('d/m/Y');
            } elseif ($isFirstPayment) {
                $payment_start_date = 'Depuis la création du compte';
            } else {
                $payment_start_date = 'Pas de paiement antérieur';
            }

            // Payment end date is the current payment's date
            $payment_end_date = Carbon::parse($payment->payment_date)->format('d/m/Y');

            // Pass the necessary information to the view
            $pdf = Pdf::loadView('receipts.payment', [
                'teacher' => $teacher,
                'payment' => $payment,
                'company_name' => env('COMPANY_NAME'),
                'company_email' => env('COMPANY_EMAIL'),
                'company_website' => env('CAMPANY_WEBSITE'),
                'payment_start_date' => $payment_start_date,
                'payment_end_date' => $payment_end_date,
            ]);

            return $pdf->download("receipt_teacher_payment_{$payment->id}.pdf");
        } catch (\Exception $e) {
            Log::error('Teacher payment receipt generation failed: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to generate receipt: ' . $e->getMessage()], 500);
        }
    }


    public function index()
    {
        $payments = TeacherPayment::query()
                        ->with(['teacher:id,first_name,last_name'])
                        ->orderBy('updated_at', 'desc')
                        ->get()->map(function ($payment) {
                            return [
                                'id' => $payment->id,
                                'teacher_id' => $payment->teacher_id,
                                'payment_method' => $payment->payment_method,
                                'teacher_name' => $payment->teacher->first_name.' '.$payment->teacher->last_name,
                                'amount' => $payment->amount,
                                'description' => $payment->description,
                                'status' => $payment->status,
                                'payment_date' => $payment->payment_date,
                                'account_name' => $payment->account_name,
                                'account_iban' => $payment->account_iban,
                                'account_bic' => $payment->account_bic,
                                'mobile_number' => $payment->mobile_number,
                                'mobile_operator' => $payment->mobile_operator,
                            ];
                        });

        return response()->json([
            'payments' => $payments
        ]);
    }

    public function show(TeacherPayment $payment)
    {
        return response()->json([
            'payment' =>  $this->formatPayment($payment),
        ]);
    }

    public function store(Request $request)
    {
        $validatedData = $request->validate([
            'teacher_id' => 'required|exists:teachers,id',
            'amount' => 'required',
            'payment_method' => 'required|string',
            'description' => 'nullable|string',
            'account_name' => 'nullable|string',
            'account_iban' => 'nullable|string',
            'account_bic' => 'nullable|string',
            'mobile_number' => 'nullable|string',
            'mobile_operator' => 'nullable|string',
            'payment_date' => 'required|date',
            'status' => "required|string"
        ]);

        $teacher = Teacher::findOrFail($validatedData['teacher_id']);

        if ($teacher->balance < $validatedData['amount']) {
            return response()->json([
                'message_code' => 'INSUFFICIENT_BALANCE',
                'message' => 'Teacher does not have sufficient balance for this payment.',
            ], 400);
        }

        $payment = TeacherPayment::create($validatedData);
         // Deduct the amount regardless of status (pending or pay)
        $teacher->balance -= $validatedData['amount'];
        $teacher->save();

        return response()->json([
            'message_code' => 'TEACHER_PAYMENT_SUCCESS',
            'payment' => $this->formatPayment($payment)
        ]);
    }

    public function update(Request $request, TeacherPayment $payment)
    {
        $validatedData = $request->validate([
            'teacher_id' => 'sometimes|required|exists:teachers,id',
            'amount' => 'sometimes|required|numeric',
            'payment_method' => 'sometimes|required|string',
            'description' => 'nullable|string',
            'account_name' => 'nullable|string',
            'account_iban' => 'nullable|string',
            'account_bic' => 'nullable|string',
            'mobile_number' => 'nullable|string',
            'mobile_operator' => 'nullable|string',
            'payment_date' => 'sometimes|required|date',
            'status' => 'sometimes|required|string'
        ]);

        $oldAmount = $payment->amount;

        // If the teacher_id is being updated, ensure the new teacher has sufficient balance
        if (isset($validatedData['teacher_id']) && $validatedData['teacher_id'] != $payment->teacher_id) {
            $newTeacher = Teacher::findOrFail($validatedData['teacher_id']);
            $newAmount = $validatedData['amount'] ?? $oldAmount;

            if ($newTeacher->balance < $newAmount) {
                return response()->json([
                    'message_code' => 'INSUFFICIENT_BALANCE',
                    'message' => 'New teacher does not have sufficient balance for this payment.',
                ], 400);
            }
            // If changing teacher, transfer the balance
            $payment->teacher->balance += $oldAmount;
            $payment->teacher->save();
            $newTeacher->balance -= $newAmount;
            $payment->teacher()->associate($newTeacher);
            $payment->teacher->save();
        } 
        else if (isset($validatedData['amount']) && $validatedData['amount'] != $oldAmount) {
            $amountDifference = $validatedData['amount'] - $oldAmount;
            if ($payment->teacher->balance < $amountDifference) {
               return response()->json([
                'message_code' => 'INSUFFICIENT_BALANCE',
                'message' => 'Teacher does not have sufficient balance for this payment.',
            ], 400);
            }
            $payment->teacher->balance -= $amountDifference;
            $payment->teacher->save();
        }
    
        $payment->update($validatedData);

        return response()->json([
            'message_code' => 'TEACHER_PAYMENT_UPDATED',
            'payment' => $this->formatPayment($payment),
        ]);
    }

    public function destroy( TeacherPayment $payment)
    {
        $payment->delete();

        return response()->json([
            'message_code' => 'TEACHER_PAYMENT_DELETED',
            'message' => 'Payment deleted successfully!',
        ]);
    }

    public function statusUpdate(Request $request, TeacherPayment $payment) 
    {
        $validatedData = $request->validate([
            'status' => 'required|in:pay,canceled'
        ]);

        $payment->status = $validatedData['status'];
        $payment->save();

        if($validatedData['status'] === 'canceled') {
            $payment->teacher->balance += $payment->amount;
            $payment->teacher->save();
        }

        return response()->json([
            'message_code' => 'TEACHER_PAYMENT_STATUS_'.strtoupper($validatedData['status']),
        ]);
    }

    private function formatPayment(TeacherPayment $payment)
    {
        return [
            'id' => $payment->id,
            'teacher_id' => $payment->teacher_id,
            'teacher_name' => $payment->teacher->first_name . ' ' . $payment->teacher->last_name,
            'amount' => $payment->amount,
            'payment_method' => $payment->payment_method,
            'description' => $payment->description,
            'status' => $payment->status,
            'payment_date' => $payment->payment_date,
            'account_name' => $payment->account_name,
            'account_iban' => $payment->account_iban,
            'account_bic' => $payment->account_bic,
            'mobile_number' => $payment->mobile_number,
            'mobile_operator' => $payment->mobile_operator,
        ];
    }
}
