<?php

namespace App\Http\Controllers;

use App\Models\Lesson;
use App\Models\Student;
use App\Models\Teacher;
use App\Models\TeacherPayment;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class TeacherController extends Controller
{
    public function index()
    {
        $teachers = Teacher::with(['courses:id,course_title', 'user:id,timezone'])->orderBy('updated_at', 'desc')->get()
            ->map(function ($teacher) {
                return [
                    'id' => $teacher->id,
                    'user_id' => $teacher->user_id,
                    'name' => $teacher->first_name.' '.$teacher->last_name,
                    'email' => $teacher->email,
                    'phone_number' => $teacher->phone_number,
                    'status' => $teacher->status,
                    'availabilities' => $teacher->availabilities,
                    'timezone' => $teacher->user ? $teacher->user->timezone : null,
                    'profile_image' => $teacher->profile_image ? asset('storage/'.$teacher->profile_image) : null,
                    'created_at' => $teacher->created_at ? $teacher->created_at : null,
                    'balance' => (double) $teacher->balance
                ];
            });

        return response()->json(['teachers' => $teachers]);
    }

    public function store(Request $request)
    {
        try {
            DB::beginTransaction();

            $validatedData = $request->validate([
                'first_name' => 'nullable|string',
                'last_name' => 'required|string',
                'email' => 'required|email|unique:teachers',
                'phone_number' => 'required|string',
            ]);

            $user = (new UserController())->register($request);

            $validatedData = array_merge($validatedData, ['user_id' => $user->id]);
            $teacher = Teacher::create($validatedData);

            // $password = $request->input('password');

            // $user->notify(new WelcomeTeacherNotification($password));

            DB::commit();

            $teacherData = [
                'id' => $teacher->id,
                'user_id' => $teacher->user_id,
                'name' => $teacher->first_name.' '.$teacher->last_name,
                'email' => $teacher->email,
                'phone_number' => $teacher->phone_number,
                'timezone' => $teacher->user ? $teacher->user->timezone : null,
                'profile_image' => $teacher->profile_image ? asset('storage/'.$teacher->profile_image) : null,
                'created_at' => $teacher->created_at ? $teacher->created_at : null,
            ];

            return response()->json([
                'teacher' => $teacherData,
                'message' => 'Teacher created successfully! Welcome email sent.',
                'message_code' => 'TEACHER_CREATE_SUCCESS',
            ]);
        } catch (\Exception $e) {
            // Something went wrong, rollback the transaction
            DB::rollback();

            return response()->json([
                'message' => 'Error creating teacher profile: '.$e->getMessage(),
            ], 500);
        }
    }

    public function show(Teacher $teacher)
    {
        $teacher->load(['user:id,timezone', 'payments']);
        
        if ($teacher->status !== 'deleted') {
            $teacher->account_pause_histories = $teacher->user->accountPauseHistories()
                                                        ->orderBy('deactivation_date', 'desc')
                                                        ->get();
            $teacher->current_pause = $teacher->user->accountPauseHistories()->whereNull('reactivation_date')->first();
        } else {
            $teacher->account_pause_histories = [];
            $teacher->current_pause = null;
        }

        $profileImageUrl = $teacher->profile_image ? asset('storage/'.$teacher->profile_image) : null;
        
        // Call the showCourses method to get the courses data
        $teacher->courses = $this->showCourses($teacher)->original['courses'];

        return response()->json(['teacher' => array_merge($teacher->toArray(), ['profile_image' => $profileImageUrl])]);
    }

    public function showLessons(Teacher $teacher)
    {
        $lessons = Lesson::where('teacher_id', $teacher->id)
            ->with([
                'course:id,course_title',
                'student:id,first_name,last_name,profile_image',
            ])
            ->get();

        $formattedLessons = $lessons->map(function ($lesson) {
            return [
                'id' => $lesson->id,
                'date' => $lesson->date,
                'time' => $lesson->time,
                'description' => $lesson->description,
                'status' => $lesson->status,
                'title' => $lesson->lesson_title,
                'course_title' => $lesson->course->course_title,
                'student_id' => $lesson->student_id,
                'student_name' => $lesson->student->first_name.' '.$lesson->student->last_name,
                'student_profile_image' => $lesson->student->profile_image ? asset('storage/'.$lesson->student->profile_image) : null,
            ];
        });

        return response()->json(['lessons' => $formattedLessons]);
    }

    public function showStudents(Teacher $teacher)
    {
        $students = Student::select('id', 'first_name', 'last_name', 'email', 'phone_number', 'status', 'academic_level_class_id', 'profile_image', 'user_id','country')
            ->with('academicLevelClass:id,class_name', 'user:id,timezone')
            ->whereHas('lessons', function ($query) use ($teacher) {
                $query->where('teacher_id', $teacher->id);
            })
            ->get();

        $studentData = $students->map(function ($student) use ($teacher) {
            $courses = Lesson::where('teacher_id', $teacher->id)
                ->where('student_id', $student->id)
                ->with('course:id,course_title')
                ->get()
                ->pluck('course')
                ->unique('id')
                ->map(function ($course) {
                    return [
                        'id' => $course->id,
                        'course_title' => $course->course_title,
                    ];
                })
                ->values()
                ->all();

            return [
                'id' => $student->id,
                'user_id' => $student->user_id,
                'name' => $student->first_name.' '.$student->last_name,
                'email' => $student->email,
                'phone' => $student->phone_number,
                'status' => $student->status,
                'class' => $student->academicLevelClass->class_name,
                'timezone' => $student->user ? $student->user->timezone ?? 'UTC' : 'UTC',
                'country' => $teacher->country,
                'profile_image' => $student->profile_image ? asset('storage/'.$student->profile_image) : null,
                'courses' => $courses,
            ];
        })->all();

        return response()->json(['students' => $studentData]);
    }

    public function getLessonsByTeacherAndStudent(Teacher $teacher, Student $student)
    {
        $lessons = Lesson::where('teacher_id', $teacher->id)
            ->where('student_id', $student->id)
            ->get();

        // Format lesson data if needed (optional)
        $formattedLessons = $lessons->map(function ($lesson) {
            return [
                'id' => $lesson->id,
                'date' => $lesson->date,
                'time' => $lesson->time,
                'description' => $lesson->description,
                'status' => $lesson->status,
                'title' => $lesson->lesson_title,
            ];
        });

        return response()->json(['lessons' => $formattedLessons]);
    }

    public function teacherCourseLink(Request $request, Teacher $teacher)
    {
        $validatedData = $request->validate([
            'course_ids' => 'required|array',
            'course_ids.*' => 'required|exists:courses,id',
        ]);

        $courseIds = $validatedData['course_ids'];
        $existingLinks = $teacher->courses()->whereIn('course_id', $courseIds)->get();

        if ($existingLinks->isNotEmpty()) {
            $linkedCourseTitles = $existingLinks->pluck('course_title')->toArray();

            return response()->json([
                'message' => 'One or more courses are already linked to the teacher.',
                'linked_courses' => $linkedCourseTitles,
                'message_code' => 'USER_COURSE_LINK_EXIST',
            ], 400);
        }

        $teacher->courses()->attach($courseIds);

        return response()->json([
            'message' => 'Teacher linked with courses successfully',
            'message_code' => 'USER_COURSE_LINK_SUCCESS',
        ]);
    }

    public function teacherCourseUnlink(Request $request, Teacher $teacher)
    {
        $validatedData = $request->validate([
            'course_ids' => 'required|array',
            'course_ids.*' => 'required|exists:courses,id',
        ]);

        $courseIds = $validatedData['course_ids'];
        $existingLinks = $teacher->courses()->whereIn('course_id', $courseIds)->get();

        if ($existingLinks->isEmpty()) {
            return response()->json([
                'message' => 'One or more courses are not linked to the teacher',
                'message_code' => 'USER_COURSE_LINK_NOT_EXIST',
            ], 400);
        }

        $teacher->courses()->detach($courseIds);

        return response()->json([
            'message' => 'Teacher unlinked with courses successfully',
            'message_code' => 'USER_COURSE_UNLINK_SUCCESS',
        ]);
    }

    public function update(Request $request, Teacher $teacher)
    {
        $validatedData = $request->validate([
            'first_name' => 'nullable|string',
            'last_name' => 'nullable|string',
            'phone_number' => 'nullable|string',
            'biography' => 'nullable|string',
            'education' => 'nullable|string',
            'experience' => 'nullable|integer',
            'age' => 'nullable|integer',
            'gender' => 'nullable|string',
            'address' => 'nullable|string',
            'profile_image' => [
                'nullable',
                'image',
                'mimes:jpeg,png,jpg',
                'max:2048',
                'dimensions:min_width=200,min_height=200,max_width=1500,max_height=1500'
            ],
            'country' => 'nullable|string',
            'city' => 'nullable|string',
            'interests' => 'nullable|string',
            'postal_code' => 'nullable|string',
            'language' => 'nullable|string',
            'availabilities' => 'nullable|string',
            'actual_job' => 'nullable|string',
        ]);

        if ($request->hasFile('profile_image')) {
            // Delete the old profile photo if it exists
            if ($teacher->profile_image) {
                Storage::disk('public')->delete($teacher->profile_image);
            }

            // Generate a unique file name
            $fileName = Str::uuid().'.'.$request->file('profile_image')->getClientOriginalExtension();

            // Store the new profile photo
            $path = $request->file('profile_image')->storeAs('images/profile_photos', $fileName, 'public');
            $validatedData['profile_image'] = $path;
        }

        if ($request->has('courses')) {
            $validatedTeacherCources = $request->validate([
                'courses' => 'required|array',
                'courses.*' => 'exists:courses,id',
            ]);
            $validatedCourses = $validatedTeacherCources['courses'];

            $teacher->courses()->sync($validatedCourses);
        }
        if ($request->has('timezone')) {
            (new UserController())->updateTimezone($request, $teacher->user);
        }

        $teacher->update($validatedData);
        $teacher->load(['courses']);
        // Generate the full URL for the profile image
        $profileImageUrl = $teacher->profile_image ? asset('storage/'.$teacher->profile_image) : null;

        return response()->json([
            'completion_rate' => $this->calculateProfileCompletionRate($teacher),
            'teacher' => array_merge($teacher->toArray(), ['profile_image' => $profileImageUrl]),
            'message' => 'Profile updated successfully!',
            'message_code' => 'PROFILE_UPDATE_SUCCESS',
        ]);
    }

    public function calculateProfileCompletionRate(Teacher $teacher)
    {
        // Define relevant fields for completion calculation
        $relevantFields = [
            'first_name',
            'last_name',
            'phone_number',
            'biography',
            'education',
            'experience',
            'age',
            'gender',
            'address',
            'profile_image',
            'country',
            'city',
            'interests',
            'postal_code',
            'actual_job',
        ];

        // Count completed fields
        $completedFields = 0;
        foreach ($relevantFields as $field) {
            if (!empty($teacher->{$field})) {
                ++$completedFields;
            }
        }

        // Calculate completion percentage
        $completionRate = round(($completedFields / count($relevantFields)) * 100);

        return $completionRate;
    }

    public function statusUpdate(Request $request, Teacher $teacher)
    {
        $validatedData = $request->validate([
            'status' => 'required|string',
        ]);

        $teacher->update($validatedData);

        return response()->json([
            'message' => 'teacher status updated successfully!',
            'message_code' => 'USER_STATUS_'.strtoupper($validatedData['status']),
        ]);
    }

    public function teacherStatusUpdate(Request $request, Teacher $teacher)
    {
        $validatedData = $request->validate([
            'status' => 'required|string',
        ]);

        $teacher->update($validatedData);
        $teacher->user->update(['status' => $validatedData['status']]);

        return response()->json([
            'message' => 'status updated successfully!',
            'message_code' => 'USER_STATUS_'.strtoupper($validatedData['status']),
        ]);
    }

    public function destroy(User $teacher)
    {
        $teacher->delete();

        return response()->json([
            'message' => 'Teacher deleted successfully!',
        ]);
    }

    public function showCourses(Teacher $teacher)
    {
        $courses = $teacher->courses()->get();

        $formattedCourses = $courses->map(function ($course) {
            return [
                'id' => $course->id,
                'course_title' => $course->course_title,
                'lesson_count' => $course->lessons()
                    ->where('teacher_id', $course->pivot->teacher_id)
                    ->where('status', 'completed')
                    ->count()
            ];
        })->all();

        return response()->json(['courses' => $formattedCourses]);
    }

    public function showCourseStudents(Teacher $teacher)
    {
        $courses = $teacher->courses()
                ->with(['students' => function ($query) use ($teacher) {
                    $query->whereHas('lessons', function ($lessonQuery) use ($teacher) {
                        $lessonQuery->where('teacher_id', $teacher->id);
                    });
                }])
                ->get();

        $students = Student::with(['user:id,timezone'])
            ->whereHas('lessons', function ($lessonQuery) use ($teacher) {
                $lessonQuery->where('teacher_id', $teacher->id);
            })->get()
            ->map(function ($student) {
                return [
                    'id' => $student->id,
                    'name' => $student->first_name . ' ' . $student->last_name,
                    'timezone' => $student->user ? $student->user->timezone ?? 'UTC' : 'UTC',
                    'courses' => $student->courses()->pluck('courses.id')->all()
                ];
            });

        $result = [
            'courses' => $courses->map(function ($course) {
                return [
                    'id' => $course->id,
                    'course_title' => $course->course_title,
                    'students' => $course->students->pluck('id')->all()
                ];
            }),
            'students' => $students
        ];

        return response()->json($result);
    }

    public function teacherStatistics(Teacher $teacher)
    {
        // Get the number of distinct students
        $numberOfStudents = $teacher->lessons()
            ->distinct('student_id')
            ->count('student_id');

        // Get the total number of lessons
        $numberOfLessons = $teacher->lessons()->count();

        $numberOfLessonsDone = $teacher->lessons()
            ->where('status', 'completed') // Assuming there's a 'status' field
            ->count();

        // Get the number of reprogrammed lessons
        $numberOfReprogrammedLessons = $teacher->lessons()
            ->where('is_reprogrammed', true)
            ->count();

        // Define period ranges
        $weeklyRange = [now()->subDays(7), now()];
        $monthlyRange = [now()->subDays(30), now()];
        $halfYearlyRange = [now()->subMonths(6), now()];
        $yearlyRange = [now()->subMonths(12), now()];

        // Get lessons statistics for the specified periods
        $weeklyStats = $this->getLessonStatsForPeriod($teacher, $weeklyRange, 'daily');
        $monthlyStats = $this->getLessonStatsForPeriod($teacher, $monthlyRange, 'daily');
        $halfYearlyStats = $this->getLessonStatsForPeriod($teacher, $halfYearlyRange, 'monthly');
        $yearlyStats = $this->getLessonStatsForPeriod($teacher, $yearlyRange, 'monthly');

        return response()->json([
            'total_students' => $numberOfStudents,
            'total_lessons' => $numberOfLessons,
            'total_lessons_done' => $numberOfLessonsDone,
            'total_lessons_reprogrammed' => $numberOfReprogrammedLessons,
            'weekly_stats' => $weeklyStats,
            'monthly_stats' => $monthlyStats,
            'half_yearly_stats' => $halfYearlyStats,
            'yearly_stats' => $yearlyStats,
        ]);
    }

    private function getLessonStatsForPeriod($teacher, $range, $groupBy)
    {
        $lessons = $teacher->lessons()
            ->whereBetween('date', $range)
            ->get();

        $grouped = [];
        $startDate = Carbon::parse($range[0]);
        $endDate = Carbon::parse($range[1]);

        // Initialize the period with zero values
        while ($startDate->lessThanOrEqualTo($endDate)) {
            $key = $groupBy === 'daily' ? $startDate->format('Y-m-d') : $startDate->format('Y-m');
            $grouped[$key] = [
                'total_scheduled' => 0,
                'total_done' => 0,
                'total_reprogrammed' => 0,
            ];
            $groupBy === 'daily' ? $startDate->addDay() : $startDate->addMonth();
        }

        // Sum lessons within the period
        foreach ($lessons as $lesson) {
            $lessonDate = Carbon::parse($lesson->date);
            $key = $groupBy === 'daily' ? $lessonDate->format('Y-m-d') : $lessonDate->format('Y-m');
            if (isset($grouped[$key])) {
                ++$grouped[$key]['total_scheduled'];
                if ($lesson->status == 'completed') {
                    ++$grouped[$key]['total_done'];
                }
                if ($lesson->is_reprogrammed) {
                    ++$grouped[$key]['total_reprogrammed'];
                }
            }
        }

        // Convert to formatted grouped array
        $formattedGrouped = [];
        foreach ($grouped as $key => $values) {
            $date = Carbon::createFromFormat($groupBy === 'daily' ? 'Y-m-d' : 'Y-m', $key);
            $formattedKey = $groupBy === 'daily' ? $date->format('d M') : $date->format('M Y');
            $formattedGrouped[$formattedKey] = $values;
        }

        return $formattedGrouped;
    }

    public function teacherNewStudentStatistics(Teacher $teacher)
    {
        // Define period ranges
        $weeklyRange = [now()->subDays(7), now()];
        $monthlyRange = [now()->subDays(30), now()];
        $halfYearlyRange = [now()->subMonths(6), now()];
        $yearlyRange = [now()->subMonths(12), now()];

        // Get first lesson statistics for the specified periods
        $weeklyFirstLessonStats = $this->getFirstLessonStatsForPeriod($teacher, $weeklyRange, 'daily');
        $monthlyFirstLessonStats = $this->getFirstLessonStatsForPeriod($teacher, $monthlyRange, 'daily');
        $halfYearlyFirstLessonStats = $this->getFirstLessonStatsForPeriod($teacher, $halfYearlyRange, 'monthly');
        $yearlyFirstLessonStats = $this->getFirstLessonStatsForPeriod($teacher, $yearlyRange, 'monthly');

        return response()->json([
            'weekly_stats' => $weeklyFirstLessonStats,
            'monthly_stats' => $monthlyFirstLessonStats,
            'half_yearly_stats' => $halfYearlyFirstLessonStats,
            'yearly_stats' => $yearlyFirstLessonStats,
        ]);
    }

    private function getFirstLessonStatsForPeriod($teacher, $range, $groupBy)
    {
        $lessons = $teacher->lessons()
            ->whereBetween('date', $range)
            ->orderBy('date')
            ->get();

        $firstLessonStudents = [];
        $grouped = [];
        $startDate = Carbon::parse($range[0]);
        $endDate = Carbon::parse($range[1]);

        // Initialize the period with zero values
        while ($startDate->lessThanOrEqualTo($endDate)) {
            $key = $groupBy === 'daily' ? $startDate->format('Y-m-d') : $startDate->format('Y-m');
            $grouped[$key] = 0;
            $groupBy === 'daily' ? $startDate->addDay() : $startDate->addMonth();
        }

        // Find first lesson dates for each student
        foreach ($lessons as $lesson) {
            $studentId = $lesson->student_id;
            if (!isset($firstLessonStudents[$studentId])) {
                $firstLessonStudents[$studentId] = $lesson->date;
            }
        }

        // Group first lessons by the specified period
        foreach ($firstLessonStudents as $date) {
            $lessonDate = Carbon::parse($date);
            $key = $groupBy === 'daily' ? $lessonDate->format('Y-m-d') : $lessonDate->format('Y-m');
            if (isset($grouped[$key])) {
                ++$grouped[$key];
            }
        }

        // Convert to formatted grouped array
        $formattedGrouped = [];
        foreach ($grouped as $key => $value) {
            $date = Carbon::createFromFormat($groupBy === 'daily' ? 'Y-m-d' : 'Y-m', $key);
            $formattedKey = $groupBy === 'daily' ? $date->format('d M') : $date->format('F Y');
            $formattedGrouped[$formattedKey] = $value;
        }

        return $formattedGrouped;
    }

    public function getTeacherStudentStatistics(Teacher $teacher, Student $student)
    {
    }

    public function showStudent(Teacher $teacher, Student $student)
    {
        $student->load(['academicLevelClass:id,class_name', 'user:id,timezone']);
        $courses = Lesson::where('teacher_id', $teacher->id)
                ->where('student_id', $student->id)
                ->with('course:id,course_title')
                ->get()
                ->pluck('course')
                ->unique('id')
                ->map(function ($course) {
                    return [
                        'id' => $course->id,
                        'course_title' => $course->course_title,
                    ];
                })
                ->values()
                ->all();
        // Get the total number of lessons given by the teacher to the student
        $totalLessons = $teacher->lessons()
            ->where('student_id', $student->id)
            ->count();

        // Get the total number of upcoming lessons between the teacher and the student
        $totalUpcomingLessons = $teacher->lessons()
            ->where('student_id', $student->id)
            ->where('date', '>', now())
            ->count();

        // Get the total number of completed lessons between the teacher and the student
        $totalCompletedLessons = $teacher->lessons()
            ->where('student_id', $student->id)
            ->where('status', 'completed') // Assuming there's a 'status' field
            ->count();

        // Get the student's credit (assuming you have a 'credits' field in the Student model)
        $studentCredit = (new StudentController())->getCredits($student);

        $student = [
            'id' => $student->id,
            'user_id' => $student->user_id,
            'name' => $student->first_name.' '.$student->last_name,
            'country' => $student->country,
            'city' => $student->city,
            'gender' => $student->gender,
            'age' => $student->age,
            'learning_objective' => $student->learning_objective,
            'interests' => $student->interests,
            'status' => $student->status,
            'class' => $student->academicLevelClass->class_name,
            'timezone' => $student->user ? $student->user->timezone ?? 'UTC' : 'UTC',
            'profile_image' => $student->profile_image ? asset('storage/'.$student->profile_image) : null,
            'courses' => $courses,
            'total_lessons' => $totalLessons,
            'total_upcoming_lessons' => $totalUpcomingLessons,
            'total_completed_lessons' => $totalCompletedLessons,
            'student_credit' => $studentCredit,
        ];

        // Return the results as a JSON response
        return response()->json([
            'student' => $student,
        ]);
    }

    public function filterTeachers(Request $request)
    {
        $validatedData = $request->validate([
            'course_id' => 'required|exists:courses,id',
            'date' => 'required|date',
            'start_time' => 'required|date_format:H:i:s',
            'lesson_id' => 'nullable|exists:lessons,id',
            'lesson_duration' => 'required'
        ]);

        $date = Carbon::parse($validatedData['date'])->toDateString();
        $startTime = Carbon::parse($validatedData['date'] . ' ' . $validatedData['start_time']);
        $endTime = $startTime->copy()->addMinutes($validatedData['lesson_duration']);
        $currentLesson = null;

        if (isset($validatedData['lesson_id'])) {
            $currentLesson = Lesson::findOrFail($validatedData['lesson_id']);
        }

        // Query for all teachers who teach the course
        $teachers = Teacher::query()
            ->whereHas('courses', function ($query) use ($validatedData) {
                $query->where('courses.id', $validatedData['course_id']);
            })->get();

        $availableTeachers = collect();

        foreach ($teachers as $teacher) {
            // Check if teacher has availability
            $hasAvailability = $teacher->availabilities()
                ->where('date', $date)
                ->where('start_time', '<=', $startTime)
                ->where('end_time', '>=', $endTime)
                ->exists();

            // Check for lesson overlaps
            $hasOverlap = $teacher->lessons()
                ->whereDate('date', $date)
                ->get()
                ->contains(function ($lesson) use ($startTime, $endTime, $currentLesson) {
                    // Skip if this is the current lesson being updated
                    if ($currentLesson && $lesson->id === $currentLesson->id) {
                        return false;
                    }

                    $lessonStartTime = Carbon::parse($lesson->date . ' ' . $lesson->time);
                    $lessonEndTime = $lessonStartTime->copy()->addMinutes($lesson->duration);

                    // Add buffer time
                    $lessonStartWithBuffer = $lessonStartTime->copy()->subMinutes(10);
                    $lessonEndWithBuffer = $lessonEndTime->copy()->addMinutes(10);

                    // Check if there's an overlap
                    return ($startTime < $lessonEndWithBuffer && $endTime > $lessonStartWithBuffer);
                });

            if (!$hasOverlap) {
                $availableTeachers->push([
                    'id' => $teacher->id,
                    'name' => $teacher->first_name . ' ' . $teacher->last_name,
                    'timezone' => $teacher->user->timezone,
                    'age' => $teacher->age,
                    'gender' => $teacher->gender,
                    'has_availability' => $hasAvailability,
                ]);
            }
        }

        return response()->json([
            'available_teachers' => $availableTeachers,
        ]);
    }

    public function getBalance()
    {
        $teachers = Teacher::orderBy('updated_at', 'desc')->get()
            ->map(function ($teacher) {
                return [
                    'id' => $teacher->id,
                    'user_id' => $teacher->user_id,
                    'name' => $teacher->first_name.' '.$teacher->last_name,
                    'balance' => (double) $teacher->balance
                ];
            });

        return response()->json(['teachers' => $teachers]);
    }

}
