<?php

namespace App\Http\Controllers;

use App\Models\Teacher;
use DateTime;
use DatePeriod;
use DateInterval;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use App\Models\TeacherAvailability;

class TeacherAvailabilityController extends Controller
{
    public function index(Teacher $teacher)
    {
        $availabilities = TeacherAvailability::where('teacher_id', $teacher->id)->orderBy('updated_at', 'desc')->get();
        return response()->json([
            'availabilities' => $availabilities
        ]);
    }

    public function store(Request $request, Teacher $teacher)
    {
        $validated = $request->validate([
            'startTimes' => 'required|array',
            'endTimes' => 'required|array',
            'startTimes.*' => 'required|date',
            'endTimes.*' => 'required|date',
        ]);

        $startTimes = $validated['startTimes'];
        $endTimes = $validated['endTimes'];

        if (count($startTimes) !== count($endTimes)) {
            return response()->json([
                'message' => 'starttimes and endtimes arrays must have the same length'
            ], 422);
        }

        $availabilities = [];
        foreach ($startTimes as $i => $start) {
            $startDateTime = new DateTime($start);
            $endDateTime = new DateTime($endTimes[$i]);

            $availabilities[] = TeacherAvailability::create([
                'teacher_id' => $teacher->id,
                'date' => $startDateTime->format('Y-m-d'),
                'start_time' => $startDateTime->format('H:i:s'),
                'end_time' => $endDateTime->format('H:i:s'),
            ]);
        }

        return response()->json([
            'availabilities' => $availabilities,
            'message' => 'Availability saved successfully'
        ]);
    }

    public function update(Request $request, Teacher $teacher, TeacherAvailability $availability)
    {
        $validated = $request->validate([
            'date' => 'required|date',
            'startTime' => 'required|date_format:H:i',
            'endTime' => 'required|date_format:H:i',
        ]);

        $availability->update([
            'date' => $validated['date'],
            'start_time' => $validated['startTime'],
            'end_time' => $validated['endTime'],
        ]);

        return response()->json([
            'availability' => $availability,
            'message' => 'Availability updated successfully'
        ]);
    }

    public function destroy(Teacher $teacher, TeacherAvailability $availability)
    {
        $availability->delete();

        return response()->json([
            'message' => 'Availability deleted successfully'
        ]);
    }
}
