<?php

namespace App\Http\Controllers;

use App\Models\SubscriptionPlan;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;

class SubscriptionPlanController extends Controller
{
    public function index()
    {
        $subscriptionPlans = SubscriptionPlan::orderBy('created_at', 'asc')->get();

        return response()->json(['subscription_plans' => $subscriptionPlans]);
    }

    public function show(SubscriptionPlan $subscriptionPlan)
    {
        return response()->json(['subscription_plan' => $subscriptionPlan]);
    }

    public function store(Request $request)
    {
        $validatedData = $request->validate([
            'name' => 'required|string|max:255',
            'credits' => 'required|integer',
            'duration_days' => 'required|integer',
            'price' => 'required|numeric',
            'description' => 'nullable|string',
        ]);

        $subscriptionPlan = SubscriptionPlan::create($validatedData);
        $subscriptionPlan->status = 'active';

        return response()->json([
            'plan' => $subscriptionPlan,
            'message' => 'Subscription plan created successfully',
            'message_code' => 'SUBSCRIPTION_PLAN_CREATE_SUCCESS',
        ]);
    }

    public function update(Request $request, SubscriptionPlan $subscriptionPlan)
    {
        $validatedData = $request->validate([
            'name' => 'nullable|string',
            'credits' => 'nullable|integer',
            'duration_days' => 'nullable|integer',
            'price' => 'nullable|numeric',
            'description' => 'nullable|string',
            'status' => 'nullable|string',
        ]);

        $subscriptionPlan->update($validatedData);

        return response()->json([
            'message' => 'Subscription plan updated successfully',
            'message_code' => 'SUBSCRIPTION_PLAN_UPDATE_SUCCESS',
        ]);
    }

    public function destroy(SubscriptionPlan $subscriptionPlan)
    {
        $subscriptionPlan->delete();

        return response()->json([
            'message' => 'Subscription plan deleted successfully',
            'message_code' => 'SUBSCRIPTION_PLAN_DELETE_SUCCESS',
        ]);
    }

    public function statusUpdate(Request $request, SubscriptionPlan $subscriptionPlan)
    {
        $validatedData = $request->validate([
            'status' => 'required|string',
        ]);

        $subscriptionPlan->update($validatedData);

        return response()->json([
            'message' => 'subscription plan status updated successfully!',
            'message_code' => 'SUBSCRIPTION_PLAN_STATUS_'.strtoupper($validatedData['status']),
        ]);
    }

    public function showSubscriptions(SubscriptionPlan $subscriptionPlan)
    {
        $subscriptions = $subscriptionPlan
                        ->subscriptions()
                        ->with('student')
                        ->orderBy('created_at', 'asc')
                        ->get();

        $formattedSubscriptions = $subscriptions->map(function ($subscription) {
            return [
                'id' => $subscription->id,
                'student_name' => $subscription->student->first_name.' '.$subscription->student->last_name,
                'credits_remaining' => $subscription->credits_remaining,
                'student_id' => $subscription->student_id,
                'status' => $subscription->status,
                'start_date' => $subscription->start_date,
            ];
        });

        return response()->json(['subscriptions' => $formattedSubscriptions]);
    }
}
