<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use Stripe\Stripe;
use App\Models\User;
use App\Models\Payment;
use App\Models\Student;
use Stripe\PaymentIntent;
use App\Events\UpdateCredit;
use App\Models\Subscription;
use Illuminate\Http\Request;
use App\Models\PaymentMethod;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Models\SubscriptionPlan;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use App\Http\Controllers\UserController;
use App\Notifications\SubscriptionNotification;
use App\Http\Controllers\PaymentMethodController;

class SubscriptionController extends Controller
{
    public function index(Request $request)
    {
        Stripe::setApiKey(env('STRIPE_SECRET'));

        $query = Subscription::with(['student:id,first_name,last_name', 'subscriptionPlan:id,name,price'])
                                        ->select(['id', 'student_id', 'subscription_plan_id', 'credits_remaining', 'status', 'end_date', 'payment_method_id'])
                                        ->orderBy('start_date', 'desc');
        if ($limit = $request->query->get('_limit')) {
            $query->limit($limit);
        }
        $subscriptions = $query->get();

        $formattedSubscriptions = $subscriptions->map(function ($subscription) {
            $student = $subscription->student;
            $subscriptionPlan = $subscription->subscriptionPlan;
            $paymentMethod = $subscription->payment?->paymentMethod;

            return [
                'id' => $subscription->id,
                'name' => $student->first_name.' '.$student->last_name,
                'student_id' => $student->id,
                'subscription_plan_id' => $subscriptionPlan->id,
                'subscription_plan_name' => $subscriptionPlan->name,
                'credits_remaining' => $subscription->credits_remaining,
                'status' => $subscription->status,
                'end_date' => $subscription->end_date,
                'payment_method_name' => $paymentMethod ? $paymentMethod->type : null,
            ];
        });

        return response()->json(['subscriptions' => $formattedSubscriptions]);
    }

     public function store(Request $request)
     {

         $validatedData = $request->validate([
            'paymentIntent_id' => 'required',
            'student_id' => 'required|exists:students,id',
            'subscription_plan_id' => 'required|exists:subscription_plans,id',
            'amount' => 'required|integer',
            'free_lesson' => 'required|boolean',
            'credits_remaining' => 'required|integer',
            'auto_renew' => 'required|boolean',
         ]);

         try { 
            DB::beginTransaction();
            $subscription = Subscription::create([
                'student_id' => $validatedData['student_id'],
                'subscription_plan_id'=> $validatedData['subscription_plan_id'],
                'credits_remaining' =>   $validatedData['credits_remaining'],
                'start_date' => Carbon::now('UTC')->toDateTimeString(),
                'end_date' => Carbon::now('UTC')->toDateTimeString(),
                'status' => 'pending',
                'auto_renew' => $validatedData['auto_renew'],
                'includes_free_lesson' => $validatedData['free_lesson'],
            ]);

           Payment::create([
                'subscription_id' => $subscription->id,
                'payment_date' => Carbon::now('UTC')->toDateTimeString(),
                'stripe_payment_id' => $validatedData['paymentIntent_id'],
                'stripe_fees'=> '0',
                'amount' => $validatedData['amount']
            ]);
            DB::commit();

            return response()->json([
                'subscription' => $subscription,
            ]);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json(['error' => $e->getMessage()], 400);
        }
     }


    public function subscriptionWithOutPayment(Request $request)
    {
        try{
            $validatedData = $request->validate([
                'student_id' => 'required|exists:students,id',
                'subscription_plan_id' => 'required|exists:subscription_plans,id',
            ]);

            $plan = SubscriptionPlan::where('id', $validatedData['subscription_plan_id'])->where('status', 'active')->first();
            $student = Student::find($validatedData['student_id']);
            $startDate = Carbon::now('UTC')->toDateTimeString();
            $endDate = null;
            $creditsRemaining = 0;

            $existingSubscription = Subscription::where('student_id', $student->id)->where('status', 'active')->first();
            if ($existingSubscription) {
                $creditsRemaining = $plan->credits + $existingSubscription->credits_remaining;
                $remainingDuration = Carbon::parse($existingSubscription->end_date)->diffInDays(Carbon::now('UTC'));
                $endDate = Carbon::parse($startDate)->addDays($plan->duration_days + $remainingDuration)->toDateTimeString();

                $existingSubscription->update([
                    'status' => 'finished',
                    'credits_remaining' => 0,
                    'end_date' => Carbon::now('UTC')->toDateTimeString(), // update the end_date to the current time
                ]);
            } else {
                $creditsRemaining = $plan->credits;
                $endDate = Carbon::parse($startDate)->addDays($plan->duration_days)->toDateTimeString();
            }

            $subscription = Subscription::create([
                'student_id' => $student->id,
                'subscription_plan_id' => $plan->id,
                'status' => 'active',
                'credits_remaining' => $creditsRemaining,
                'start_date' => $startDate,
                'end_date' => $endDate,
                'auto_renew' => false,
                'payment_method_id' => null,
                'created_by' => auth()->id(),
            ]);

            $subscriptionData = [
                'id' => $subscription->id,
                'name' => $student->first_name.' '.$student->last_name,
                'student_id' => $student->id,
                'subscription_plan_id' => $plan->id,
                'subscription_plan_name' => $plan->name,
                'credits_remaining' => $subscription->credits_remaining,
                'status' => $subscription->status,
                'end_date' => $subscription->end_date,
                'payment_method_name' => null,
            ];

            return response()->json([
                'subscription' => $subscriptionData,
                'message' => 'Subscription created successfully!',
            ]);
        } catch (\Exception $e) {
            Log::error('Error creating supscription without payment method: ' . $e->getMessage());
            return response()->json(['error' => 'Error creating supscription without payment method: ' . $e->getMessage()], 400);
        }
    }

    public function show(Subscription $subscription)
    {
        $subscription->load(['student', 'subscriptionPlan', 'payment']);
        $paymentMethod = $subscription->payment?->paymentMethod()->first();
        $endDate = Carbon::parse($subscription->end_date);
        $renewalDate = $subscription->auto_renew ? $endDate->copy()->addDay() : null;
        $createdBy = User::find($subscription->created_by);
        $paymentMethodInfo = null;

        if ($paymentMethod) {
            if ($paymentMethod->type === 'card') {
                $paymentMethodInfo = $paymentMethod->card_brand.'...'.$paymentMethod->card_last4;
            } else {
                $paymentMethodInfo = $paymentMethod->type;
            }
        }

        return response()->json([
            'subscription' => [
                'id' => $subscription->id,
                'plan_name' => $subscription->subscriptionPlan->name,
                'price' => $subscription->subscriptionPlan->price,
                'credits_remaining' => $subscription->credits_remaining,
                'start_date' => $subscription->start_date,
                'end_date' => $subscription->end_date,
                'renewal_date' => $renewalDate ? $renewalDate : null,
                'auto_renew' => $subscription->auto_renew,
                'status' => $subscription->status,
                'created_by' => $createdBy && $createdBy->hasRole('admin') ? 'admin' : null,
                'student' => [
                    'name' => $subscription->student->first_name.' '.$subscription->student->last_name,
                    'email' => $subscription->student->email,
                    'phone' => $subscription->student->phone_number,
                ],
                'payment_method' => $paymentMethodInfo,
            ],
        ]);
    }

    public function update(Request $request, Subscription $subscription)
    {
        $validatedData = $request->validate([
            'status' => 'nullable|string|in:finished,active,canceled,expired,suspended,pending',
            'credits_remaining' => 'nullable|integer',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date',
            'auto_renew' => 'nullable|boolean',
        ]);

        try {
            DB::beginTransaction();
            $createdBy = User::find($subscription->created_by);

            if ($request->has('auto_renew') && $createdBy && $createdBy->hasRole('admin')) {
                return response()->json([
                    'message' => "Auto renew for admin created subcriptions is not possible.",
                    'message_code' => 'SUBSCRIPTION_ADMIN_RESTRICTED'
                ], 400);
            }

            $subscription->update($validatedData);

            if (User::find(auth()->user()->id)->hasRole('admin')) {
                $student = Student::find($subscription->student_id);
                $student->user->notify(new SubscriptionNotification($subscription, 'updated', $student->user->id));
            }

            DB::commit();
            return response()->json([
                'subscription' => $subscription,
                'message' => 'Subscription updated successfully!',
                'message_code' => 'SUBSCRIPTION_UPDATE_SUCCESS',
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Subscription update failed: ' . $e->getMessage());
            return response()->json([
                'message' => 'An error occurred while updating the subscription.',
                'message_code' => 'SUBSCRIPTION_UPDATE_FAILED',
                'error' => $e->getMessage()
            ], 400);
        }
    }

    public function statusUpdate(Request $request, Subscription $subscription)
    {
        $validatedData = $request->validate([
            'status' => 'required|string',
        ]);
        $messagecode = null;

        try {
            DB::beginTransaction();
            $student = Student::find($subscription->student_id);

            if ($validatedData['status'] === 'suspended') {
                $subscription->suspended_at = Carbon::now('UTC')->toDateTimeString();
                $subscription->save();
                $messagecode = 'SUBSCRIPTION_STATUS_SUSPENDED';
                $student->user->notify(new SubscriptionNotification($subscription, 'suspended', $student->user->id));
            } elseif ($validatedData['status'] === 'active' && $subscription->status === 'suspended') {
                $suspendedAt = Carbon::parse($subscription->suspended_at);
                $endDate = Carbon::parse($subscription->end_date);
                $daysSuspended = max($suspendedAt->diffInDays($endDate, false) + 1, 1);
                $subscription->end_date = Carbon::now('UTC')->addDays($daysSuspended)->toDateTimeString();
                $subscription->suspended_at = null; // Clear the suspended_at field
                $subscription->save();
                $messagecode = 'SUBSCRIPTION_STATUS_REACTIVATED';
                $student->user->notify(new SubscriptionNotification($subscription, 'reactivation_after_suspension', $student->user->id));
            }

            $subscription->update($validatedData);
            DB::commit();

            return response()->json([
                'subscription' => $subscription,
                'message' => 'Subscription status updated successfully!',
                'message_code' => $messagecode,
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Subscription status update failed: ' . $e->getMessage());
            return response()->json([
                'message' => 'An error occurred while updating the subscription status.',
                'error' => $e->getMessage()
            ], 400);
        }
    }

    public function destroy(Subscription $subscription)
    {
        $subscription->delete();

        return response()->json([
            'message' => 'Subscription deleted successfully!',
        ]);
    }

    public function cancel(Subscription $subscription)
    {
        try {
            DB::beginTransaction();
            $subscription->end_date = Carbon::now('UTC')->toDateTimeString();
            $subscription->status = 'canceled';
            $subscription->credits_remaining = 0;
            $subscription->save();

            $student = Student::find($subscription->student_id);
            $student->user->notify(new SubscriptionNotification($subscription, 'canceled', $student->user->id));
    
            // Trigger credit update event
            event(new UpdateCredit($student->user->id));
            DB::commit();
            return response()->json([
                'subscription' => $subscription,
                'message' => 'Subscription canceled successfully!',
                'message_code' => 'SUBSCRIPTION_CANCEL_SUCCESS',
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Subscription manual cancellation failed: ' . $e->getMessage());
    
            return response()->json([
                'message' => 'Une erreur est survenue lors de l\'annulation de l\'abonnement.',
                'status' => 'error'
            ], 400);
        }
    }

    public function completed(Request $request, Subscription $subscription)
    {
        Stripe::setApiKey(env('STRIPE_SECRET'));
        try {
            $validatedData = $request->validate([
                'paymentIntent_id' => 'required',
                'payment_method_id' => 'required',
            ]);

            // Perform the completed action
            $subscription = $this->completedAction($subscription, $validatedData['paymentIntent_id'], $validatedData['payment_method_id']);

            if($subscription->includes_free_lesson){
                $admins = User::role('admin')->get();
                foreach ($admins as $admin) {
                    $admin->notify(new SubscriptionNotification($subscription, 'subscription_with_lesson', $admin->id));
                }
            }

            return response()->json([
                'subscription' => $subscription,
                'message_code' => 'SUBSCRIPTION_CREATION_SUCCESS',
            ]);
        } catch (\Exception $e) {
            Log::error('An error occurred while processing the payment: ' . $e->getMessage());
            return response()->json([
                'message_code' => 'SUBSCRIPTION_PAYMENT_FAILED',
                'message' => 'An error occurred while processing the payment. Please try again.'
            ], 400);
        }
    }

    public function SavePaymentMethodAfterSubscription(Request $request, Subscription $subscription)
    {
        Stripe::setApiKey(env('STRIPE_SECRET'));
        try {
            $validatedData = $request->validate([
                'paymentIntent_id' => 'required|string',
                'payment_method_id' => 'required|string',
            ]);

       
            $user = User::find(Auth::user()->id);
            $paymentMethods = $user->paymentMethods()->count();
            if ($paymentMethods === 3) {
                return response()->json([
                    'message_code' => 'MAX_PAYMENT_METHOD_SAVE',
                    'message' => 'You have reached the maximum number of payment methods allowed.'
                ], 400);
            }

            $paymentMethod = PaymentMethod::where('stripe_payment_method_id', $validatedData['payment_method_id'])->first();

            if (!$paymentMethod) {
                $data = (new PaymentMethodController())->storeAction(
                    paymentMethodId: $validatedData['payment_method_id'],
                    chargeId: PaymentIntent::retrieve($validatedData['paymentIntent_id'])->latest_charge,
                );
                if (isset($data['error'])) {
                    return response()->json([
                        'message_code' => 'PAYMENT_METHOD_SAVE_FAILED',
                        'data' => $data
                    ], 400);
                } elseif (isset($data['paymentMethod'])) {
                    $paymentMethod = $data['paymentMethod'];
                    $paymentMethod->status = 'active';
                    $paymentMethod->save();
                } else {
                    // If no paymentMethod was found or created, return an error
                    return response()->json([
                        'message_code' => 'PAYMENT_METHOD_SAVE_FAILED',
                        'message' => 'Failed to retrieve or create a payment method.',
                        'data' => $data,
                    ], 400);
                }
            }

            return response()->json([
                'subscription' => $subscription,
                'message' => 'Payment method saved successfully!',
                'message_code' => 'PAYMENT_METHOD_ADD_SUCCESS',
            ],200);
        } catch (\Exception $e) {
            Log::error('An error occurred while saving the payment method: ' . $e->getMessage());
            return response()->json([
                'message_code' => 'PAYMENT_METHOD_SAVE_FAILED',
                'message' => 'An error occurred while saving the payment method. Please try again.'
            ], 400);
        }
    }

    public function completedAction(Subscription $subscription, $paymentIntentId, $paymentMethodId)
    {
        $payment = $subscription->payment;
        $existingSubscription = Subscription::where('student_id', $subscription->student->id)->where('status', 'active')->first();
        $remainingDuration = 0;
        $creditsRemaining = 0;

        if ($existingSubscription) {
            $remainingDuration = Carbon::parse($existingSubscription->end_date)->diffInDays(Carbon::now('UTC'));
            $creditsRemaining = $existingSubscription->credits_remaining;
            
            $existingSubscription->update([
                'status' => 'finished',
                'credits_remaining' => 0,
                'end_date' => Carbon::now('UTC')->toDateTimeString(),
            ]);
        } 

        $subscription->status = 'active';
        $subscription->start_date = Carbon::now('UTC')->toDateTimeString();
        $subscription->end_date = Carbon::now('UTC')->addDays($subscription->subscriptionPlan->duration_days + $remainingDuration);
        $subscription->credits_remaining = ($subscription->credits_remaining ?: 0) + $creditsRemaining;
        $subscription->save();

        $payment->status = 'completed';
        $payment->stripe_payment_method_id = $paymentMethodId;
        $payment->stripe_payment_id = $paymentIntentId;
        $payment->save();

        return $subscription;
    }

    public function failed(?Request $request = null, Subscription $subscription)
    {
        $validatedData = $request->validate([
            'paymentIntent_id' => 'required',
            'payment_method_id' => 'required',
        ]);

        $this->failedAction($subscription, $validatedData['paymentIntent_id'], $validatedData['payment_method_id']);

        return response()->json([
            'message' => 'Subscription failed,  please try again.',
        ]);
    }

    public function failedAction(Subscription $subscription, $paymentIntentId = null, $paymentMethodId): void
    {
        $subscription->start_date = Carbon::now('UTC')->toDateTimeString();
        $subscription->end_date = Carbon::now('UTC')->toDateTimeString();
        $subscription->status = 'canceled';
        $subscription->payment_method_id = $paymentMethodId;
        $subscription->credits_remaining = 0;
        $subscription->save();

        $payment = $subscription->payment;
        $payment->status = 'failed';
        $payment->stripe_payment_id = $paymentIntentId;
        $payment->stripe_payment_method_id = $paymentMethodId;
        $payment->save();
    }

    public function subscriptionReceipt(Subscription $subscription)
    {
        try {
            // Fetch related details
            $subscription->load(['student', 'payment.paymentMethod', 'subscriptionPlan']);

            // Format the data for the receipt
            $subscriptionData = [
                'student_name' => $subscription->student->first_name . ' ' . $subscription->student->last_name,
                'student_email' => $subscription->student->email,
                'company_name' => env('COMPANY_NAME'),
                'company_email' => env('COMPANY_EMAIL'),
                'company_website' => env('CAMPANY_WEBSITE'),
                'subscription_plan_name' => $subscription->subscriptionPlan->name,
                'subscription_amount' => number_format($subscription->subscriptionPlan->price, 2, ',', ' ') . ' EUR',
                'subscription_start_date' => Carbon::parse($subscription->start_date)->format('d/m/Y'),
                'subscription_end_date' => Carbon::parse($subscription->end_date)->format('d/m/Y'),
                'payment_date' => $subscription->payment ? Carbon::parse($subscription->payment->payment_date)->format('d/m/Y') : 'N/A',
                'payment_method' => $subscription->payment && $subscription->payment->paymentMethod 
                                    ? ($subscription->payment->paymentMethod->type === 'card'
                                        ? $subscription->payment->paymentMethod->card_brand . ' : .... .... ....' . $subscription->payment->paymentMethod->card_last4
                                        : $subscription->payment->paymentMethod->type)
                                    : 'N/A',
            ];

            // Generate the PDF using the receipt template
            $pdf = Pdf::loadView('receipts.subscription', $subscriptionData);

            return $pdf->download("subscription_receipt_{$subscription->id}.pdf");
        } catch (\Exception $e) {
            Log::error('Receipt generation failed: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to generate receipt: ' . $e->getMessage()], 400);
        }
    }

}
