<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use Stripe\Charge;
use Stripe\Stripe;
use App\Models\User;
use App\Models\Lesson;
use App\Models\Payment;
use App\Models\Student;
use App\Models\Teacher;
use Stripe\PaymentIntent;
use Illuminate\Support\Str;
use App\Models\Subscription;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Notifications\UserNotification;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Artisan;
use App\Notifications\SubscriptionNotification;
use App\Jobs\ProcessAutoRenewalJob;

class StudentController extends Controller
{
    public function index()
    {
        $students = Student::with(['courses:id,course_title', 'academicLevelClass:id,class_name,academic_level_id', 'academicLevelClass.academicLevel:id,level_title', 'user:id,timezone'])->orderBy('updated_at', 'desc')->get()
                            ->map(function ($student) {
                                return [
                                    'id' => $student->id,
                                    'user_id' => $student->user_id,
                                    'name' => $student->first_name.' '.$student->last_name,
                                    'email' => $student->email,
                                    'phone_number' => $student->phone_number,
                                    'status' => $student->status,
                                    'class' => $student->academicLevelClass ? $student->academicLevelClass->class_name : null,
                                    'level' => $student->academicLevelClass->academicLevel ? $student->academicLevelClass->academicLevel->level_title : null,
                                    'timezone' => $student->user ? $student->user->timezone : null,
                                    'profile_image' => $student->profile_image ? asset('storage/'.$student->profile_image) : null,
                                    'created_at' => $student->created_at ? $student->created_at : null,
                                ];
                            });

        return response()->json(['students' => $students]);
    }

    public function store(Request $request)
    {
        $validatedData = $request->validate([
            'academic_level_class_id' => 'required|exists:academic_level_classes,id',
            'email' => 'required|email|unique:students,email',
            'first_name' => 'required|string',
            'last_name' => 'required|string',
            'phone_number' => 'required|string',
            'learning_objective' => 'nullable|string',
            'course_id' => 'required|exists:courses,id',
            'user'
        ]);

        $courseId = $validatedData['course_id'];
        unset($validatedData['course_id']);

        DB::beginTransaction();
        try {
            $user = (new UserController())->register($request);
            $studentData = array_merge($validatedData, ['user_id' => $user->id]);
            $student = Student::create($studentData);

            $student->courses()->attach($courseId);
            
            $student->load(['courses', 'academicLevelClass']);

            $loginResponse = (new UserController())->login($request); // Log in the user
            DB::commit();

            $studentName = $student->first_name.' '.$student->last_name;
            $admins = User::role('admin')->get(); 
            foreach ($admins as $admin) {
                $admin->notify(new UserNotification($student->id, $studentName, 'new_account', 'student',null));
            }
            
            return response()->json([
                'student' => json_decode($loginResponse->getContent()),
                'message' => 'Student created successfully!',
                'message_code' => 'ACCOUNT_CREATED_SUCCESS',
            ]);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'error' => 'Failed to create the student.',
                'message' => $e->getMessage(),
            ]);
        }
    }

    public function show(Student $student)
    {
        $student->load(['academicLevelClass:id,class_name,academic_level_id', 'academicLevelClass.academicLevel:id,level_title', 'user:id,timezone']);

        if ($student->status !== 'deleted') {
            $student->account_pause_histories = $student->user->accountPauseHistories()->orderBy('deactivation_date', 'desc')->get();
            $student->current_pause = $student->user->accountPauseHistories()->whereNull('reactivation_date')->first();
        } else {
            $student->account_pause_histories = [];
            $student->current_pause = null;
        }

        $profileImageUrl = $student->profile_image ? asset('storage/'.$student->profile_image) : null;

        // Call the showCourses method to get the courses data
        $student->courses = $this->showCourses($student)->original['courses'];

        return response()->json(['student' => array_merge($student->toArray(), ['profile_image' => $profileImageUrl])]);
    }

    public function showLessons(Student $student)
    {
        $lessons = Lesson::where('student_id', $student->id)
                            ->with([
                                'course:id,course_title', 
                                'teacher:id,first_name,last_name,profile_image',
                            ])
                            ->get();

        $formattedLessons = $lessons->map(function ($lesson) {

            return [
                'id' => $lesson->id,
                'date' => $lesson->date,
                'time' => $lesson->time,
                'description' => $lesson->description,
                'status' => $lesson->status,
                'title' => $lesson->lesson_title,
                'course_title' => $lesson->course->course_title,
                'teacher_id' => $lesson->student_id,
                'teacher_name' => $lesson->teacher ? $lesson->teacher->first_name.' '.$lesson->teacher->last_name : null,
                'teacher_profile_image' => $lesson->teacher && $lesson->teacher->profile_image ? asset('storage/'.$lesson->teacher->profile_image) : null,
            ];
        });

        return response()->json(['lessons' => $formattedLessons]);
    }

    public function showCourses(Student $student)
    {
        $courses = $student->courses()->get();

        $formattedCourses = $courses->map(function ($course) {
            return [
                'id' => $course->id,
                'course_title' => $course->course_title,
                'lesson_count' => $course->lessons()
                    ->where('student_id', $course->pivot->student_id)
                    ->where('status', 'completed')
                    ->count()
            ];
        })->all();

        return response()->json(['courses' => $formattedCourses]);
    }

    public function studentLinkCourse(Request $request, Student $student)
    {
        $validatedData = $request->validate([
            'course_ids' => 'required|array',
            'course_ids.*' => 'required|exists:courses,id',
        ]);

        $courseIds = $validatedData['course_ids'];
        $existingLinks = $student->courses()->whereIn('course_id', $courseIds)->get();

        if ($existingLinks->isNotEmpty()) {
            $linkedCourseTitles = $existingLinks->pluck('course_title')->toArray();

            return response()->json([
                'message' => 'Une ou plusieurs de ces matières sont déjà liées à votre compte.',
                'linked_courses' => $linkedCourseTitles,
                'message_code' => 'USER_COURSE_LINK_EXIST',
            ], 400);
        }

        $student->courses()->attach($courseIds);

        $courses = $this->showCourses($student)->original['courses'];

        return response()->json([
            'courses' => $courses, 
            'message' => 'La ou les matières ont été ajoutées avec succès à votre liste.',
            'message_code' => 'USER_COURSE_LINK_SUCCESS',
        ]);
    }

    public function showCourseTeachers(Student $student)
    {
        $courses = $student->courses()
            ->with(['teachers' => function ($query) use ($student) {
                $query->whereHas('lessons', function ($lessonQuery) use ($student) {
                    $lessonQuery->where('student_id', $student->id);
                });
            }])
            ->get();

        $teachers = Teacher::with(['user:id,timezone'])
            ->whereHas('lessons', function ($query) use ($student) {
                $query->where('student_id', $student->id);
            })
            ->get()
            ->map(function ($teacher) {
                return [
                    'id' => $teacher->id,
                    'name' => $teacher->first_name . ' ' . $teacher->last_name,
                    'timezone' => $teacher->user ? $teacher->user->timezone ?? 'UTC' : 'UTC',
                    'courses' => $teacher->courses()->pluck('courses.id')->all()
                ];
            });

        $result = [
            'courses' => $courses->map(function ($course) {
                return [
                    'id' => $course->id,
                    'course_title' => $course->course_title,
                    'teachers' => $course->teachers->pluck('id')->all()
                ];
            }),
            'teachers' => $teachers
        ];

        return response()->json($result);
    }
    public function studentCourseUnLink(Request $request, Student $student)
    {
        $validatedData = $request->validate([
            'course_ids' => 'required|array',
            'course_ids.*' => 'required|exists:courses,id',
        ]);

        $courseIds = $validatedData['course_ids'];
        $existingLinks = $student->courses()->whereIn('course_id', $courseIds)->get();

        if ($existingLinks->isEmpty()) {
            return response()->json([
                'message' => 'Une ou plusieurs de ces matières ne sont pas liés à votre compte.',
                'message_code' => 'USER_COURSE_LINK_NOT_EXIST',
            ], 400);
        }

        $student->courses()->detach($courseIds);
        $courses = $this->showCourses($student)->original['courses'];

        return response()->json([
            'courses' => $courses, 
            'message' => 'La matière a été retirer de votre liste avec succès.',
            'message_code' => 'USER_COURSE_UNLINK_SUCCESS',
        ]);
    }

    public function showTeachers(Student $student)
    {
        // Eager load lessons, students, and levels with courses and pivot table
        $lessons = $student->lessons()->with([
            'teacher' => function ($query) {
                $query->select('id', 'first_name', 'last_name', 'email', 'status', 'user_id', 'phone_number', 'profile_image','country');
                $query->with('courses.teachers'); // Include pivot table eagerness
                $query->orderBy('created_at', 'asc'); // Order lessons for first lesson
            },
        ])->get();

        $teacherData = $lessons->pluck('teacher')->filter(function ($teacher) {
            return !is_null($teacher);
        })->unique('id')->values()->map(function ($teacher) {
            $courses = $teacher->courses->all();

            return [
                'id' => $teacher->id,
                'user_id' => $teacher->user_id,
                'name' => $teacher->first_name.' '.$teacher->last_name,
                'email' => $teacher->email,
                'phone_number' => $teacher->phone_number,
                'status' => $teacher->status,
                'country' => $teacher->country,
                'timezone' => $teacher->user ? $teacher->user->timezone ?? 'UTC' : 'UTC',
                'courses' => $courses, // Include fetched 
                'profile_image' => $teacher->profile_image ? asset('storage/'.$teacher->profile_image) : null,
            ];
        })->all();

        return response()->json(['teachers' => $teacherData]);
    }

    public function getStudentSubscriptions(Student $student)
    {
        Stripe::setApiKey(env('STRIPE_SECRET'));
        $subscriptions = Subscription::where('student_id', $student->id)
                        ->where('status', '!=', 'canceled')
                        ->select('id', 'subscription_plan_id', 'start_date','end_date','status')
                        ->with(['subscriptionPlan:id,name,price,duration_days,credits'])
                        ->orderBy('created_at', 'desc')
                        ->get();

        $formattedSubscriptions = $subscriptions->map(function ($subscription) {
            if($subscription->subscriptionPlan && $subscription->subscriptionPlan->price){
                $amount = $subscription->subscriptionPlan->price * $subscription->subscriptionPlan->credits;
            }
            $subscriptionData = [
                'id' => $subscription->id,
                'subscription_plan_name' => $subscription->subscriptionPlan->name ?? null,
                'amount' => $amount ?? null,
                'start_date' => $subscription->start_date,
                'status' => $subscription->status,
                'end_date' => $subscription->end_date,
                'payment_date' => $subscription->payment?->payment_date,
            ];
    
             // Determine the payment method name
             if ($subscription->payment && $subscription->payment->paymentMethod) {
                $paymentMethod = $subscription->payment->paymentMethod;

                $subscriptionData['payment_method_name'] = $paymentMethod->type === 'card'
                    ? $paymentMethod->card_brand . ' : .... .... .... ' . $paymentMethod->card_last4
                    : $paymentMethod->type;
            } else {
                // Fetch from Stripe if local payment method is missing
                if ($subscription->payment && $subscription->payment->stripe_payment_method_id) {
                    $stripePaymentMethod = \Stripe\PaymentMethod::retrieve($subscription->payment->stripe_payment_method_id);
                    $subscriptionData['payment_method_name'] = $stripePaymentMethod->card
                        ? $stripePaymentMethod->card->brand . ' : .... .... .... ' . $stripePaymentMethod->card->last4
                        : $stripePaymentMethod->type;
                } else {
                    $subscriptionData['payment_method_name'] = 'N/A'; // Default value
                }
            }
    
            return $subscriptionData;
        });

        return response()->json(['subscriptions' => $formattedSubscriptions]);
    }

    public function processPendingSubscriptionsForStudent(Student $student)
    {
        Stripe::setApiKey(env('STRIPE_SECRET'));
        $status = null;
        // Fetch subscriptions with status 'pending' for the given student
        $pendingSubscriptions = Subscription::where('status', 'pending')
            ->where('student_id', $student->id)
            ->get();

        foreach ($pendingSubscriptions as $subscription) {
            $payment = Payment::where('subscription_id', $subscription->id)
                ->where('status', 'pending')
                ->first();

            if ($payment) {
                try {
                    // Retrieve the payment intent from Stripe
                    $paymentIntent = PaymentIntent::retrieve($payment->stripe_payment_id);
                    $statusToHandle = $intent->metadata->status_to_handle ?? null;

                    if ($statusToHandle !== 'processing') {
                        switch ($paymentIntent->status) {
                            case 'requires_payment_method':
                                $status = 'requires_payment_method';
                                $subscription->delete();
                                break;

                            case 'failed':
                                $status = 'failed';

                                // Mark subscription and payment as failed
                                $payment->status = 'failed';
                                $payment->stripe_payment_id = $paymentIntent->id;
                                $payment->stripe_payment_method_id = $paymentIntent->payment_method;
                                $payment->save();

                                $subscription->start_date = Carbon::now('UTC')->toDateTimeString();
                                $subscription->end_date = Carbon::now('UTC')->toDateTimeString();
                                $subscription->status = 'canceled';
                                $subscription->payment_method_id = null;
                                $subscription->credits_remaining = 0;
                                $subscription->save();

                                $student->user->notify(new SubscriptionNotification($subscription, 'subscription_payment_failed', $student->user->id));
                                break;

                            case 'succeeded':
                                $status = 'succeeded';
                                // Check if metadata is empty
                                $payment->status = 'completed';
                                $payment->stripe_payment_id = $paymentIntent->id;
                                $payment->stripe_payment_method_id = $paymentIntent->payment_method;
                                $payment->save();

                                $existingSubscription = Subscription::where('student_id', $subscription->student->id)->where('status', 'active')->orderBy('created_at', 'desc')->first();
                                $remainingDuration = 0;
                                $creditsRemaining = 0;

                                if ($existingSubscription) {
                                    $remainingDuration = Carbon::parse($existingSubscription->end_date)->diffInDays(Carbon::now('UTC'));
                                    $creditsRemaining = $existingSubscription->credits_remaining;
                                    $existingSubscription->update([
                                        'status' => 'finished',
                                        'credits_remaining' => 0,
                                        'end_date' => Carbon::now('UTC')->toDateTimeString(),
                                    ]);
                                }

                                $subscription->status = 'active';
                                $subscription->start_date = Carbon::now('UTC')->toDateTimeString();
                                $subscription->end_date = Carbon::now('UTC')->addDays($subscription->subscriptionPlan->duration_days + $remainingDuration);
                                $subscription->credits_remaining = ($subscription->credits_remaining ?: 0) + $creditsRemaining;
                                $subscription->save();
                                $student->user->notify(new SubscriptionNotification($subscription, 'activated', $student->user->id));

                                if($subscription->includes_free_lesson){
                                    $admins = User::role('admin')->get();
                                    foreach ($admins as $admin) {
                                        $admin->notify(new SubscriptionNotification($subscription, 'subscription_with_lesson', $admin->id));
                                    }
                                }
                            break;

                            default:
                                // Handle other statuses if needed
                                break;
                        }
                    }
                } catch (\Exception $e) {
                    // Log or handle Stripe API errors
                    Log::error("Error processing subscription ID {$subscription->id}: {$e->getMessage()}");
                }
            }
        }

        return response()->json([
            'message' => 'Pending subscriptions processed for the student successfully.',
            'status' => $status,
        ]);
    }

    public function getCredits(Student $student)
    {
        $latestSubscription = Subscription::where('student_id', $student->id)
                                            ->where('status', 'active')
                                            ->orderByDesc('created_at')
                                            ->first();

        if ($latestSubscription) {
            // If there is an active subscription, return the credits remaining
            return $latestSubscription->credits_remaining;
        } else {
            return 0;
        }
    }

    public function getCurrentSubscriptionstatus(Student $student)
    {
        $latestSubscription = Subscription::where('student_id', $student->id)
                                            ->whereIn('status', ['expired', 'active', 'finished', 'suspended'])  
                                            ->orderByDesc('created_at')
                                            ->first();

        return $latestSubscription ? $latestSubscription->status : null;
    }

    public function deductCredits(Student $student, $amount)
    {
        $currentCredits = $this->getCredits($student);
        $newCredits = max(0, $currentCredits - $amount);

        $latestSubscription = Subscription::where('student_id', $student->id)
            ->where('status', 'active')
            ->orderByDesc('created_at')
            ->first();

        if ($latestSubscription) {
            $latestSubscription->credits_remaining = $newCredits;

            if ($newCredits == 0) {
                $latestSubscription->status = 'finished';
                $latestSubscription->end_date = Carbon::now('UTC')->toDateTimeString();
                $student->user->notify(new SubscriptionNotification($latestSubscription, 'finished', $student->user->id));

                if ($latestSubscription->auto_renew) {
                      // Run the command asynchronously
                      ProcessAutoRenewalJob::dispatch($latestSubscription);
                }
            }

            $latestSubscription->save();
        }
    }

    public function getCurrentSubscription(Student $student)
    {
        $latestSubscription = Subscription::where('student_id', $student->id)
                                            ->whereIn('status', ['expired', 'active', 'finished']) 
                                            ->with(['subscriptionPlan:id,name,price,credits'])
                                            ->orderBy('created_at', 'desc') 
                                            ->first();
        $createdBy = $latestSubscription ? ($latestSubscription->created_by ? User::find($latestSubscription->created_by) : null) : null;

        return response()->json([
            'subscription' => $latestSubscription ? [
                'id' => $latestSubscription->id,
                'subscription_plan' => $latestSubscription->subscriptionPlan,
                'credits_remaining' => $latestSubscription->credits_remaining,
                'end_date' => $latestSubscription->end_date,
                'auto_renew' => $latestSubscription->auto_renew,
                'is_admin_created' => $createdBy && $createdBy->hasRole('admin'),
            ] : null,
     ]);
    }

    public function update(Request $request, Student $student)
    {
        $validatedData = $request->validate([
            'academic_level_class_id' => 'nullable|exists:academic_level_classes,id',
            'email' => 'nullable|email|unique:students,email,'.$student->id,
            'first_name' => 'nullable|string',
            'last_name' => 'nullable|string',
            'phone_number' => 'nullable|string',
            'learning_objective' => 'nullable|string',
            'gender' => 'nullable|in:homme,femme,autre',
            'address' => 'nullable|string',
            'interests' => 'nullable|string',
            'age' => 'nullable|integer',
            'city' => 'nullable|string',
            'country' => 'nullable|string',
            'postal_code' => 'nullable|string',
            'courses' => 'nullable|array',
            'courses.*' => 'exists:courses,id',
            'profile_image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        if ($request->hasFile('profile_image')) {
            // Delete the old profile photo if it exists
            if ($student->profile_image) {
                Storage::disk('public')->delete($student->profile_image);
            }

            // Generate a unique file name
            $fileName = Str::uuid().'.'.$request->file('profile_image')->getClientOriginalExtension();

            // Store the new profile photo
            $path = $request->file('profile_image')->storeAs('images/profile_photos', $fileName, 'public');
            $validatedData['profile_image'] = $path;
        }

        if ($request->has('password')) {
            (new UserController())->updatePassword($request, $student->user);
        }

        if ($request->has('timezone')) {
            (new UserController())->updateTimezone($request, $student->user);
        }

        $student->update($validatedData);
        $student->load(['courses', 'academicLevelClass']);

        // Generate the full URL for the profile image
        $profileImageUrl = $student->profile_image ? asset('storage/'.$student->profile_image) : null;

        return response()->json([
            'student' => array_merge($student->toArray(), ['profile_image' => $profileImageUrl]),
            'message' => 'Profile updated successfully!',
            'message_code' => 'PROFILE_UPDATE_SUCCESS',
        ]);
    }

    public function statusUpdate(Request $request, Student $student)
    {
        $validatedData = $request->validate([
            'status' => 'required|string',
        ]);

        $student->update($validatedData);
        $student->user->update(['status' => $validatedData['status']]);

        return response()->json([
            'message' => 'Student status updated successfully!',
            'message_code' => 'USER_STATUS_'.strtoupper($validatedData['status']),
        ]);
    }

    public function destroy(Student $student)
    {
        $student->delete();

        return response()->json([
            'message' => 'Student deleted successfully!',
        ]);
    }

    public function checkFreeLesson(Student $student)
    {
        $haveFreeLesson = $student->free_lesson_redeemed || 
            $student->subscriptions()->where('includes_free_lesson', true)->exists();

        return response()->json([
            'haveFreeLesson' => $haveFreeLesson,
        ]);
    }

    public function studentStatistics(Student $student)
    {
        // Total number of courses
        $totalCourses = $student->courses()->get()->count();

        // Number of lessons done
        $lessonsDone = Lesson::where('student_id', $student->id)
                             ->where('status', 'completed')
                             ->count();

        // Number of lessons to do
        $lessonsToDo = Lesson::where('student_id', $student->id)
                             ->where('status', 'scheduled')
                             ->count();

        // Set Stripe API key
        Stripe::setApiKey(env('STRIPE_SECRET'));

        // Fetch payments from Stripe
        $payments = [];
        if ($student->user->stripe_id) {
            $charges = Charge::all(['customer' => $student->user->stripe_id, 'limit' => 100]);
            foreach ($charges->data as $charge) {
                $payments[] = [
                    'amount' => round($charge->amount / 100, 2), // Convert amount from cents to dollars
                    'currency' => strtoupper($charge->currency),
                    'date' => Carbon::createFromTimestamp($charge->created),
                ];
            }
        }

        // Group payments by the specified periods
        $weeklyPayments = $this->calculatePaymentsForPeriod($payments, now()->subDays(7), 'daily');
        $monthlyPayments = $this->calculatePaymentsForPeriod($payments, now()->subDays(30), 'daily');
        $halfYearlyPayments = $this->calculatePaymentsForPeriod($payments, now()->subMonths(6), 'monthly');
        $yearlyPayments = $this->calculatePaymentsForPeriod($payments, now()->subMonths(12), 'monthly');

        return response()->json([
            'total_courses' => $totalCourses,
            'lessons_done' => $lessonsDone,
            'lessons_to_do' => $lessonsToDo,
            'weekly_stats' => $weeklyPayments,
            'monthly_stats' => $monthlyPayments,
            'half_yearly_stats' => $halfYearlyPayments,
            'yearly_stats' => $yearlyPayments,
        ]);
    }

    private function calculatePaymentsForPeriod($payments, $startDate, $groupBy)
    {
        $grouped = [];
        $endDate = now();

        // Set locale to French
        setlocale(LC_TIME, 'fr_FR.UTF-8', 'fr_FR', 'fr', 'french');
        Carbon::setLocale('fr');

        // Initialize the period with zero values
        while ($startDate->lessThanOrEqualTo($endDate)) {
            $key = $groupBy === 'daily' ? $startDate->format('Y-m-d') : $startDate->format('Y-m');
            $grouped[$key] = 0;
            $groupBy === 'daily' ? $startDate->addDay() : $startDate->addMonth();
        }

        // Sum payments within the period
        foreach ($payments as $payment) {
            $key = $groupBy === 'daily' ? $payment['date']->format('Y-m-d') : $payment['date']->format('Y-m');
            if (isset($grouped[$key])) {
                $grouped[$key] += $payment['amount'];
            }
        }

        // Round all amounts to 2 decimal places
        $grouped = array_map(function ($amount) {
            return round($amount, 2);
        }, $grouped);

        // Convert keys to desired format in French
        $formattedGrouped = [];
        foreach ($grouped as $key => $value) {
            $date = Carbon::createFromFormat($groupBy === 'daily' ? 'Y-m-d' : 'Y-m', $key);
            if ($groupBy === 'daily') {
                $formattedKey = mb_convert_encoding(ucfirst($date->isoFormat('DD MMM')), 'UTF-8', 'UTF-8'); // e.g., "15 juin"
            } else {
                $formattedKey = mb_convert_encoding(ucfirst($date->isoFormat('MMMM YYYY')), 'UTF-8', 'UTF-8'); // e.g., "Juin 2023"
            }
            $formattedGrouped[$formattedKey] = $value;
        }

        return $formattedGrouped;
    }
}
