<?php

namespace App\Http\Controllers;

use App\Models\Payment;
use App\Models\Subscription;
use App\Models\SubscriptionPlan;
use App\Models\Teacher;
use App\Models\TeacherPayment;
use Illuminate\Http\JsonResponse;
use Illuminate\Routing\Controller;

class RevenueController extends Controller
{
    public function index(): JsonResponse
    {
        return response()->json([
            'student_subscription_revenue' => $this->getStudentSubscriptionRevenue(),
            'current_month_revenue' => $this->getCurrentMonthSubscriptionRevenue(),
            'total_teachers_balance' => $this->getTotalTeachersBalance(),
            'total_teachers_payment' => $this->getTotalTeachersPayment(),
            'revenue_per_subcriptions_plans' => $this->revenuePerSubcriptionPlan()
        ]);
    }

    private function revenuePerSubcriptionPlan(){
        return SubscriptionPlan::query()
        ->where("status", "active")
        ->get()
        ->reduce(function(array $carry, SubscriptionPlan $subscriptionPlan){
            array_push($carry[0], $subscriptionPlan->name);
            $revenue = $subscriptionPlan->subscriptions->reduce(function (float $revenue, Subscription $subscription) {
                return $revenue + $subscription->payment()
                ->whereIn("status", ["completed"])
                ->get()->reduce(fn (float $v, Payment $p) => $v + $p->amount, 0);
            }, 0);
            array_push($carry[1], $revenue / 100);
            return $carry;
        }, [[], []]);
    }

    private function getStudentSubscriptionRevenue()
    {
        return Payment::query()
        ->whereIn("status", ["completed"])
        ->get()
        ->sum('amount') / 100;
    }

    private function getCurrentMonthSubscriptionRevenue()
    {
        // Get the start and end of the current month
        $startOfMonth = now()->startOfMonth();
        $endOfMonth = now()->endOfMonth();

        return Payment::query()
            ->whereIn("status", ["completed"])
            ->whereBetween('created_at', [$startOfMonth, $endOfMonth]) // Filter by payment date
            ->get()
            ->sum('amount') / 100;
    }

    private function getTotalTeachersBalance(){
        return Teacher::query()
        ->get()
        ->sum('balance');
    }

    private function getTotalTeachersPayment()
    {
        return TeacherPayment::query()
        ->where('status', 'pay')
        ->get()
        ->sum('amount');
    }
}
