<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Report;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use App\Notifications\ReportedProblemNotification;

class ReportController extends Controller
{
    public function index()
    {
        $reports = Report::with(['reporter', 'reporter.teacher', 'reporter.student'])
                ->orderBy('created_at', 'desc')
                ->get();

        $formattedreports = $reports->map(function ($report) {
            $reportData = [];

            if ($report->reporter->hasRole('student')) {
                $reportData['reporter_id'] = $report->reporter->student ? $report->reporter->student->id : null;
                $reportData['reporter_name'] = $report->reporter->student ? $report->reporter->student->first_name.' '.$report->reporter->student->last_name : null;
                $reportData['reporter_role'] = 'student';
                $reportData['reporter_role_id'] = $report->reporter->student ? $report->reporter->student->id : null;
            } elseif ($report->reporter->hasRole('teacher')) {
                $reportData['reporter_id'] = $report->reporter->teacher ? $report->reporter->teacher->id : null;
                $reportData['reporter_name'] = $report->reporter->teacher ? $report->reporter->teacher->first_name.' '.$report->reporter->teacher->last_name : null;
                $reportData['reporter_role'] = 'teacher';
                $reportData['reporter_role_id'] = $report->reporter->teacher ? $report->reporter->teacher->id : null;
            } else {
                $reportData['reporter_id'] = $report->reporter->id;
                $reportData['reporter_name'] = 'Admin';
                $reportData['reporter_role'] = 'admin';
            }

            if ($report->reportable_type === 'User') {
                $reportableUser = User::find($report->reportable_id);
                $reportData['reportable_role'] = $reportableUser ? $reportableUser->getRoleNames()->first() : null;
            } else {
                $reportData['reportable_role'] = null;
            }

            return array_merge([
                'id' => $report->id,
                'report_type' => $report->report_type,
                'description' => $report->description,
                'category' => $report->category,
                'reportable_id' => $report->reportable_id,
                'reportable_type' => $report->reportable_type,
                'resolved_at' => $report->resolved_at,
                'attachment_path' => $report->attachment_path,
                'resolution_notes' => $report->resolution_notes,
                'status' => $report->status,
                'created_at' => $report->created_at,
            ], $reportData);
        });

        return response()->json(["reports" => $formattedreports]);
    }

    public function store(Request $request)
    {
        DB::beginTransaction();
        try {
            $validatedData = $request->validate([
                'report_type' => 'required|string|max:255',
                'description' => 'required|string',
                'category' => 'required|string',
                'attachment_path' => 'nullable|file|mimes:jpg,jpeg,png,pdf,doc,docx|max:5120',
                'reportable_id' => 'nullable|integer',
                'reportable_type' => 'nullable|string',
            ]);

            if ($request->hasFile('attachment_path')) {
                // Generate a unique file name
                $fileName = Str::uuid().'.'.$request->file('attachment_path')->getClientOriginalExtension();
    
                // Store the new profile photo
                $path = $request->file('attachment_path')->storeAs('docs/report_attachments', $fileName, 'public');
                $validatedData['attachment_path'] = $path;
            }

            $validatedData['reporter_id'] = Auth::id();
            $report = Report::create($validatedData);

            $admins = User::role('admin')->get(); 
            foreach ($admins as $admin) {
                $admin->notify(new ReportedProblemNotification($report, $admin->id));
            }

            DB::commit();

            return response()->json([
                'message' => 'Report successfully created.',
                'report' => $report,
                'message_code' => 'REPORT_CREATION_SUCCESS',
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'error' => 'Failed to create the report.',
                'message_code' => 'REPORT_CREATION_ERROR',
                'message' => $e->getMessage(),
            ], 400);
        }
    }

    public function statusUpdate(Request $request, Report $report)
    {
        try {
            $validatedData = $request->validate([
                'status' => 'required|string|in:pending,resolved,in_progress',
                'resolution_notes' => 'nullable|string',
            ]);

            $report->status = $validatedData['status'];
            if ($validatedData['status'] === 'resolved' && !$report->resolved_at) {
                $report->resolved_at = now();
                $report->resolution_notes = $validatedData['resolution_notes'] ?? null;
            }
            $report->save();

            return response()->json([
                'message' => 'Report status successfully updated.',
                'message_code' => 'REPORT_MARK_'.strtoupper($validatedData['status']),
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Failed to update the report status.',
                'message' => $e->getMessage(),
                'message_code' => 'REPORT_STATUS_UPDATE_ERROR',
            ], 400);
        }
    }

    public function update(Request $request, Report $report)
    {
        try {
            // Validate the request
            $validatedData = $request->validate([
                'description' => 'nullable|string',
                'category' => 'nullable|string',
            ]);

            // Update the report fields
            $report->description = $validatedData['description'];
            $report->category = $validatedData['category'];
            $report->save();

            return response()->json([
                'message' => 'Report successfully updated.',
                'report' => $report,
                'message_code' => 'REPORT_UPDATE_SUCCESS',
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Failed to update the report.',
                'message_code' => 'REPORT_UPDATE_ERROR',
                'message' => $e->getMessage(),
            ], 400);
        }
    }

    public function destroy(Report $report)
    {
        try {
            // Delete the attachment file if it exists
            if ($report->attachment_path && Storage::disk('public')->exists($report->attachment_path)) {
                Storage::disk('public')->delete($report->attachment_path);
            }

            // Delete the report
            $report->delete();

            return response()->json([
                'message' => 'Report successfully deleted.',
                'message_code' => 'REPORT_DELETION_SUCCESS',
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Failed to delete the report.',
                'message_code' => 'REPORT_DELETION_ERROR',
                'message' => $e->getMessage(),
            ], 400);
        }
    }

    public function show(Report $report)
    {
        $report->load(['reporter', 'reporter.teacher', 'reporter.student']);
    
        $reportData = [];
    
        if ($report->reporter->hasRole('student')) {
            $reportData['reporter_id'] = $report->reporter->student ? $report->reporter->student->id : null;
            $reportData['reporter_name'] = $report->reporter->student ? $report->reporter->student->first_name . ' ' . $report->reporter->student->last_name : null;
            $reportData['reporter_role'] = 'student';
            $reportData['reporter_role_id'] = $report->reporter->student ? $report->reporter->student->id : null;
        } elseif ($report->reporter->hasRole('teacher')) {
            $reportData['reporter_id'] = $report->reporter->teacher ? $report->reporter->teacher->id : null;
            $reportData['reporter_name'] = $report->reporter->teacher ? $report->reporter->teacher->first_name . ' ' . $report->reporter->teacher->last_name : null;
            $reportData['reporter_role'] = 'teacher';
            $reportData['reporter_role_id'] = $report->reporter->teacher ? $report->reporter->teacher->id : null;
        } else {
            $reportData['reporter_id'] = $report->reporter->id;
            $reportData['reporter_name'] = 'Admin';
            $reportData['reporter_role'] = 'admin';
        }

        if ($report->reportable_type === 'User') {
            $reportableUser = User::find($report->reportable_id);
            if ($reportableUser) {
                $reportData['reportable_role'] = $reportableUser->getRoleNames()->first();
                if ($reportData['reportable_role'] === 'student') {
                    $reportData['reportable_role_id'] = $reportableUser->student ? $reportableUser->student->id : null;
                } elseif ($reportData['reportable_role'] === 'teacher') {
                    $reportData['reportable_role_id'] = $reportableUser->teacher ? $reportableUser->teacher->id : null;
                } else {
                    $reportData['reportable_role_id'] = null;
                }
            } else {
            $reportData['reportable_role'] = null;
            $reportData['reportable_role_id'] = null;
            }
        } else {
            $reportData['reportable_role'] = null;
            $reportData['reportable_role_id'] = null;
        }
    
        $formattedReport = array_merge([
            'id' => $report->id,
            'report_type' => $report->report_type,
            'description' => $report->description,
            'category' => $report->category,
            'reportable_id' => $report->reportable_id,
            'reportable_type' => $report->reportable_type,
            'resolved_at' => $report->resolved_at,
            'attachment_path' => $report->attachment_path ? asset('storage/'.$report->attachment_path) : null,
            'resolution_notes' => $report->resolution_notes,
            'status' => $report->status,
            'created_at' => $report->created_at,
        ], $reportData);
    
        return response()->json([
            'report' => $formattedReport,
        ]);
    }
}
