<?php

namespace App\Http\Controllers;

use Stripe\Charge;
use Stripe\Stripe;
use App\Models\User;
use Stripe\SetupAttempt;
use Stripe\PaymentIntent;
use App\Models\Subscription;
use Illuminate\Http\Request;
use App\Models\PaymentMethod;
use Illuminate\Support\Carbon;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;

class PaymentMethodController extends Controller
{
    public function store(Request $request){
        $validatedData = $request->validate([
            'payment_method_id' => 'required|string',
            'setup_intent_id' => 'nullable|string',
            'paymentIntent_id' => 'nullable|string',
        ]);

        Stripe::setApiKey(env('STRIPE_SECRET'));

        $paymentMethodId = $validatedData['payment_method_id'];
        $setupIntentId = $validatedData['setup_intent_id'] ?? null;
        $paymentIntentId = $validatedData['paymentIntent_id'] ?? null;
        $user = User::find(Auth::user()->id);
        $paymentMethods = $user->paymentMethods()->count();
        $chargeId = null;
        $data = null;

        if ($paymentMethods === 3) {
            return response()->json([
                'message_code' => 'MAX_PAYMENT_METHOD_ADD',
                'message' => 'You have reached the maximum number of payment methods allowed.'
            ], 400);
        }
        if($paymentIntentId){
            $chargeId = \Stripe\PaymentIntent::retrieve($validatedData['paymentIntent_id'])->latest_charge;
        }
            
        try {
            $paymentMethod = PaymentMethod::where('stripe_payment_method_id', $validatedData['payment_method_id'])->first();

            if (!$paymentMethod) {
                $data = (new PaymentMethodController())->storeAction(
                    paymentMethodId: $paymentMethodId,
                    setupIntentId: $setupIntentId,
                    chargeId: $chargeId,
                );
                if (isset($data['error'])) {
                    return response()->json([
                        'message_code' => 'PAYMENT_METHOD_ADD_FAILED',
                        'data' => $data
                    ], 400);
                }
            } else {
                return response()->json([
                    'message' => 'Payment method already added!',
                ],200);
            }

            return response()->json($data);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()]);
        }
    }


    public function storeAction($paymentMethodId, $setupIntentId = null, $chargeId = null)
    {
        try {
            Stripe::setApiKey(env('STRIPE_SECRET'));
            $user = Auth::user();

                
            $paymentMethod = \Stripe\PaymentMethod::retrieve($paymentMethodId);
            $data = [
                'stripe_payment_method_id' => $paymentMethod->id,
                'type' => $paymentMethod->type,
                'user_id' => $user->id,
                'created_at' => Carbon::parse($paymentMethod->created)->toDateTimeString()
            ];

            if($paymentMethod->card){
                $data = [
                    ...$data,
                    'card_exp_month' => $paymentMethod->card->exp_month,
                    'card_exp_year'=> $paymentMethod->card->exp_year, 
                    'card_last4' =>  $paymentMethod->card->last4,
                    'card_brand' =>  $paymentMethod->card->brand,
                ];
            }else if($paymentMethod->paypal){
                $data = [
                    ...$data,
                    'paypal_payer_email' => $paymentMethod->paypal->payer_email,
                ];
            } else {

                if($setupIntentId){
                    $setupAttempts = SetupAttempt::all([
                        'setup_intent' => $setupIntentId
                    ]);

                    $data['stripe_payment_method_id']  = $setupAttempts->data[0]->payment_method_details->bancontact->generated_sepa_debit;
                } else if($chargeId){
                    $charge = Charge::retrieve($chargeId);

                    $data['stripe_payment_method_id']  = $charge->payment_method_details->bancontact->generated_sepa_debit;
                }

                $paymentMethod = $paymentMethod::retrieve($data['stripe_payment_method_id']);
                $data['billing_details_email'] = $paymentMethod->billing_details->email;
            }

            $paymentMethod = PaymentMethod::create($data);
            return [
                'paymentMethod' => $paymentMethod,
                'message' => 'New Payment Method create successfully',
                'message_code' => 'PAYMENT_METHOD_ADD_SUCCESS',
            ];

        } catch (\Exception $e) {
            // Handle other types of exceptions
            return [
                'error' => $e->getMessage(),
            ];
        }

    }

    public function detachedPmFromStripe(Request $request) {
        $validatedData = $request->validate([
            'payment_method_id' => 'required|string',
        ]);
        Stripe::setApiKey(env('STRIPE_SECRET'));

        \Stripe\PaymentMethod::retrieve($validatedData['payment_method_id'])->detach();

        return response()->json([
            'message' => 'Payment method removed from stripe successfully!',
        ],200);
    }

    public function destroy(PaymentMethod $paymentMethod)
    {
        Stripe::setApiKey(env('STRIPE_SECRET'));

        try {
            $stripePaymentMethod = \Stripe\PaymentMethod::retrieve($paymentMethod->stripe_payment_method_id);
            $stripePaymentMethod->detach();
            $paymentMethod->delete();

            return response()->json([
                'message' => 'Payment method deleted successfully!',
                'message_code' => 'PAYMENT_METHOD_DELETE_SUCCESS',
            ]);
        } catch (\Stripe\Exception\CardException $e) {
            return response()->json(['error' => 'An error occurred while deleting the payment method.'. $e->getMessage()], 400);
        } catch (\Exception $e) {
            return response()->json(['error' => 'An error occurred while deleting the payment method.'], 400);
        }
    }
}
