<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use Stripe\Stripe;
use App\Models\User;
use App\Models\Payment;
use App\Models\Student;
use Stripe\PaymentIntent;
use App\Models\Subscription;
use Illuminate\Http\Request;
use App\Models\SubscriptionPlan;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use App\Notifications\SubscriptionNotification;

class PaymentController extends Controller
{
    public function createSecret(Request $request)
    {
        $user = Auth::user();

        $validatedData = $request->validate([
            'amount' => 'required|integer',
        ]);

        Stripe::setApiKey(env('STRIPE_SECRET'));

        $customer = (new UserController())->createStripeCustomerAccount($user);

        try { // Get the amount from your frontend
            $paymentIntent = PaymentIntent::create([
                'customer' => $customer->id,
                'setup_future_usage' => 'off_session',
                'amount' => $validatedData['amount'],
                'currency' => 'EUR', // Change to your currency
            ]);

            return response()->json([
                'paymentIntent' => $paymentIntent,
            ]);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 400);
        }
    }

    public function SaveCardPayment(Request $request)
    {
        $user = Auth::user();
        $validatedData = $request->validate([
            'payment_method_id' => 'required',
            'amount' => 'required|integer',
            'student_id' => 'required|exists:students,id',
            'subscription_plan_id' => 'required|exists:subscription_plans,id',
            'free_lesson' => 'required|',
        ]);

        Stripe::setApiKey(env('STRIPE_SECRET'));
        $customerId = $user->stripe_id;

        $subscriptionPlan = SubscriptionPlan::find($validatedData['subscription_plan_id']);

        DB::beginTransaction();

        try {
            $subscription = Subscription::create([
                'student_id' => $validatedData['student_id'],
                'subscription_plan_id'=> $validatedData['subscription_plan_id'],
                'credits_remaining' =>  $subscriptionPlan->credits,
                'start_date' => Carbon::now('UTC')->toDateTimeString(),
                'end_date' => Carbon::now('UTC')->toDateTimeString(),
                'status' => 'pending',
                'includes_free_lesson' => $validatedData['free_lesson'],
            ]);

            $payment = Payment::create([
                'subscription_id' => $subscription->id,
                'stripe_payment_method_id' => $validatedData['payment_method_id'],
                'amount' => $validatedData['amount'],
                'payment_date' => Carbon::now('UTC')->toDateTimeString(),
                'stripe_fees' => '0',
            ]);

            $paymentIntent = null;
            $autoRenew = false;
            $latestSubscription = null;
            $latestSubscription = Subscription::where('student_id', $subscription->student->id)
                                                ->whereIn('status', ['expired', 'active', 'finished']) // Filter by specific statuses
                                                ->orderBy('created_at', 'desc') // Order by the latest created date
                                                ->first(); // Get the first (latest) record 
            if($latestSubscription) { $autoRenew = $latestSubscription->auto_renew ; }

            $paymentIntent = PaymentIntent::create([
                'customer' => $customerId,
                'payment_method' => $validatedData['payment_method_id'],
                'payment_method_types' => ['sepa_debit', 'paypal', 'card', 'bancontact'],
                'amount' => $validatedData['amount'],
                'currency' => 'EUR', // Change to your currency
                'off_session' => true,
                'confirm' => true,
            ]);

            $payment->payment_date = Carbon::parse($paymentIntent->created);
            $payment->stripe_payment_id = $paymentIntent->id;
            $payment->save();
            $subscription->auto_renew = $autoRenew;
            $subscription->save();

            DB::commit();

            if ($paymentIntent->status === 'succeeded') {
                $subscription = (new SubscriptionController())->completedAction($subscription, $paymentIntent->id, $paymentIntent->payment_method);
                
                if($subscription->includes_free_lesson){
                    $admins = User::role('admin')->get();
                    foreach ($admins as $admin) {
                        $admin->notify(new SubscriptionNotification($subscription, 'subscription_with_lesson', $admin->id));
                    }
                }
                return response()->json([
                'subscription' => $subscription,
                'status' => 'success',
                'message' => 'Subscription created successfully!',
                'message_code' => 'SUBSCRIPTION_CREATION_SUCCESS',
                ]);
            } elseif ($paymentIntent->status === "processing") {
                return response()->json([
                'status' => 'processing',
                'message' => 'Payment processing.',
                'message_code' => 'SUBSCRIPTION_PAYMENT_PROCESSING',
                ]);

            } elseif ($paymentIntent->status === 'requires_action') {
                $clientSecret = $paymentIntent->client_secret;

                return response()->json([
                    'status' => 'requires_action',
                    'client_secret' => $clientSecret,
                    'subscription' => $subscription, 
                    'payment_intent_id' => $paymentIntent->id,
                ]);
            } else {
                (new SubscriptionController())->failedAction($subscription, $paymentIntent->id, $validatedData['payment_method_id']);

                return response()->json([
                    'status' => 'failed',
                    'error' => 'Payment failed',
                    'subscription' => $subscription, 
                    'paymentIntent' => $paymentIntent,
                    'message_code' => 'SUBSCRIPTION_PAYMENT_FAILED',
                ], 400);
            }
        } catch (\Stripe\Exception\CardException $e) {
            DB::rollBack();
            $errorResponse = $e->getError();

             // Handle insufficient funds error
            if ($errorResponse->code === 'card_declined' && $errorResponse->decline_code === 'insufficient_funds') {
                if($subscription) {
                    (new SubscriptionController())->failedAction($subscription, $paymentIntent?->id, $validatedData['payment_method_id']);
                }
                return response()->json([
                    'status' => 'failed',
                    'error' => 'Your card has insufficient funds. Please use a different card or add funds to your current card.',
                    'message_code' => 'INSUFFICIENT_FUNDS',
                ], 400);
            }

            // Handle authentication required
            if ($errorResponse->code === 'authentication_required') {
                $paymentMethodId = $errorResponse->payment_method->id;
                $clientSecret = $errorResponse->payment_intent->client_secret;

                return response()->json([
                    'requires_action' => true,
                    'client_secret' => $clientSecret,
                    'subscription' => $subscription, 
                    'payment_method_id' => $paymentMethodId,
                ], 200);
            }

            // Handle other card errors
            if($subscription) {
                (new SubscriptionController())->failedAction($subscription, $paymentIntent?->id, $validatedData['payment_method_id']);
            }
            return response()->json([
                'status' => 'failed',
                'error' => $errorResponse->message,
                'message_code' => 'PAYMENT_METHOD_ERROR',
            ], 400);

        } catch (\Exception $e) {
            DB::rollBack();
            if($subscription){
                (new SubscriptionController())->failedAction($subscription, $paymentIntent?->id, $validatedData['payment_method_id']);
            }
            return response()->json([
                'error' => $e->getMessage(),
                'message_code' => 'SUBSCRIPTION_PAYMENT_FAILED',
            ], 400);
        }
    }

    public function transferFunds(Request $request)
    {
        $validatedData = $request->validate([
            'stripe_id' => 'required',
            'amount' => 'required|integer',
        ]);

        $stripe = new \Stripe\StripeClient(env('STRIPE_SECRET'));

        try {
            $transfer = $stripe->transfers->create([
                'amount' => $validatedData['amount'],
                'currency' => 'eur',
                'destination' => $validatedData['stripe_id'],
            ]);

            return response()->json([
                'message' => 'Bank detail added successfully!',
                'result' => $transfer,
            ]);
        } catch (\Stripe\Exception\ApiErrorException $e) {
            return response()->json(['error' => 'Failed to transfer the funds: '.$e->getMessage()], 500);
        }
    }
}
