<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Lesson;
use App\Models\Message;
use App\Models\Student;
use App\Models\Teacher;
use App\Events\MessageRead;
use App\Events\MessageSent;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use App\Notifications\NewChatMessageNotification;

class MessageController extends Controller
{
    public function index()
    {
        $messages = Lesson::orderBy('updated_at', 'desc')
                            ->get();

        return response()->json(['messages' => $messages]);
    }

    public function sendMessage(Request $request)
    {
        DB::beginTransaction();
        try {
            $validatedData = $request->validate([
                'sender_id' => 'required|exists:users,id',
                'receiver_id' => 'required|exists:users,id',
                'message' => 'required',
                'online_status' => 'required',
            ]);

            $receiverOnlineStatus = $validatedData['online_status'];
            unset($validatedData['online_status']);

            $message = Message::create($validatedData);

            $receiver = User::withTrashed()->findOrFail($validatedData['receiver_id']);
            $sender = User::findOrFail(Auth::user()->id);

            if (!$receiver->trashed()) {
                event(new MessageSent($message));
                
                $senderDetails = null;
                $receiverType = 'user';

                if($receiverOnlineStatus == false && $receiver->notificationPref->email_notifications) {
                    if ($receiver->hasRole('student')) {
                        $receiverType = 'student';
                        $senderDetails = $sender->hasRole('student') 
                            ? $sender->student 
                            : $sender;
                    } elseif ($receiver->hasRole('teacher')) {
                        $receiverType = 'teacher';
                        $senderDetails = $sender->hasRole('teacher') 
                            ? $sender->teacher 
                            : $sender;
                    } elseif ($receiver->hasRole('admin')) {
                        $receiverType = 'admin';
                        $senderDetails = $sender;
                    }

                    $receiver->notify(new NewChatMessageNotification(
                        $message, 
                        $senderDetails, 
                        $receiverType, 
                        $receiver->id
                    ));
                }
            }

            DB::commit();
            return response()->json([
                'message_status' => 'Message Sent!',
                'message' => $message,
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'error' => 'An unexpected error occurred while sending the message',
                'message' => $e->getMessage()
            ], 400);
        }
    }

    public function markMessageAsRead(Message $message)
    {
        $message->update([
            'read_at' => now(),
            'read' => true,
        ]);

        event(new MessageRead($message));

        return response()->json(['message' => 'Message marked as read successfully.']);
    }

    public function destroy(Message $message)
    {
        $message->delete();

        return response()->json([
            'message' => 'Message deleted successfully!',
        ]);
    }
}
