<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\User;
use Firebase\JWT\JWT;
use App\Models\Lesson;
use GuzzleHttp\Client;
use App\Models\Meeting;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use App\Notifications\LessonNotification;

class MeetingController extends Controller
{
    public function index()
    {
        $meetings = Meeting::all()->orderBy('updated_at', 'desc')->get();

        return response()->json(['meetings' => $meetings]);
    }

    public function createMeeting(Request $request)
    {
        $validatedData = $request->validate([
            'lesson_id' => 'required|exists:lessons,id',
        ]);
        
        $user = Auth::user();
        $lesson = Lesson::findOrFail($validatedData['lesson_id']);
        
        // Authorize access (you may use middleware for this as well)
        $this->authorizeMeetingAccess($lesson, $user);
    
        // Check if the lesson status is valid for starting a session
        if (!in_array($lesson->status, ['scheduled', 'paused', 'in_progress'])) {
            return response()->json([
                'error' => 'Lesson is '.$lesson->status,
            ], 400);
        }
    
        // Check if a meeting already exists
        if ($lesson->meeting) {
            if (($user->teacher?->id === $lesson->teacher_id && $lesson->teacher_present) ||
            ($user->student?->id === $lesson->student_id && $lesson->student_present)) {
            return response()->json(['error' => 'User already connected to the meeting'], 403);
            }
            return response()->json([
                'message' => 'Existing meeting',
                'meeting' => $lesson->meeting,
            ]);
        }
    
        // Create a new meeting via API
        try {
            $responseData = $this->createMeetingViaApi();
    
            DB::beginTransaction();
    
            try {
                $meetingData = [
                    'startDate' => $responseData['startDate'],
                    'endDate' => $responseData['endDate'],
                    'meeting_url' => $responseData['roomUrl'],
                    'roomName' => $responseData['roomName'],
                    'room_id' => $responseData['meetingId'],
                    'lesson_id' => $lesson->id,
                    'total_duration' => 0,
                ];
    
                $createdMeeting = $this->store(new Request($meetingData));
    
                DB::commit();
    
                return $createdMeeting;
            } catch (\Exception $e) {
                DB::rollBack();
                Log::error('Error storing meeting: ' . $e->getMessage());
                return response()->json(['error' => 'Error storing meeting'], 500);
            }
        } catch (\Exception $e) {
            Log::error('Error creating meeting via API: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to create meeting'], 500);
        }
    }
    

    private function createMeetingViaApi()
    {
        $apiKey = env('WHEREBY_API_KEY');
        $client = new Client();
        $endDate = now()->addMinutes(50)->toISOString();
        $requestData = [
            'endDate' => $endDate,
        ];

        try {
            $response = $client->post('https://api.whereby.dev/v1/meetings', [
                'headers' => [
                    'Authorization' => 'Bearer '.$apiKey,
                    'Content-Type' => 'application/json',
                ],
                'json' => $requestData,
            ]);

            if ($response->getStatusCode() === 201) {
                return json_decode($response->getBody(), true);
            } else {
                throw new \Exception('Failed to create meeting');
            }
        } catch (\Exception $e) {
            throw new \Exception('Error creating meeting: '.$e->getMessage());
        }
    }

    public function store(Request $request)
    {
        $validatedData = $request->validate([
            'startDate' => 'required|date',
            'endDate' => 'required|date',
            'meeting_url' => 'required|string',
            'roomName' => 'required|string',
            'room_id' => 'required|string',
            'lesson_id' => 'required|exists:lessons,id',
            'total_duration' => 'required|',
        ]);

        $meeting = Meeting::create($validatedData);

        return response()->json([
            'meeting' => $meeting,
            'message' => 'Meeting created successfully!',
        ]);
    }

    public function show(Meeting $meeting)
    {
        $meeting->load(['lesson']);

        return response()->json(['meeting' => $meeting]);
    }

    public function participantJoin(Meeting $meeting)
    {
        $user = User::find(auth()->user()->id);
        $lesson = $meeting->lesson;

        // Authorize access (you may use middleware for this as well)
        $this->authorizeMeetingAccess($lesson, $user);
        if ($user->hasRole('admin')) {
            return response()->json(['message' => 'Admin joined']);
        }

        $presentField = ($user->teacher?->id === $lesson->teacher_id) ? 'teacher_present' : 'student_present';
        if ($lesson->$presentField) {
            // Disconnect all users (implement this logic)
            return response()->json(['error' => 'User already connected'], 400);
        }

        try {
            DB::beginTransaction();

            $lesson->$presentField = true;
        
            if ($lesson->teacher_present && $lesson->student_present && 
                in_array($lesson->status, ['scheduled', 'paused'])) {
                $lesson->status = 'in_progress';
                $meeting->session_start_time = now();
                
                // Check if the lesson date and time are in the future
                $currentDateTime = Carbon::now();
                $lessonDateTime = Carbon::parse($lesson->date . ' ' . $lesson->time);
                if ($lessonDateTime->isFuture()) {
                    $lesson->date = $currentDateTime->toDateString();
                    $lesson->time = $currentDateTime->toTimeString();
                }  elseif ($lessonDateTime->diffInMinutes($currentDateTime) <= 15 && $lessonDateTime->isPast()) {
                    $lesson->date = $currentDateTime->toDateString();
                    $lesson->time = $currentDateTime->toTimeString();
                }
            }
    
            $lesson->save();
            $meeting->save();
    
            DB::commit();

            return response()->json([
            'total_duration' => $meeting->total_duration,
            'lesson_status' => $lesson->status,
            ]);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json(['message' => 'Error joining meeting: '.$e->getMessage()], 500);
        }
    }

    public function participantLeave(Meeting $meeting)
    {
        $user = User::find(auth()->user()->id);
        $lesson = $meeting->lesson;

        // Authorize access (you may use middleware for this as well)
        $this->authorizeMeetingAccess($lesson, $user);
        if ($user->hasRole('admin')) {
            return response()->json(['message' => 'Admin disconnected']);
        }
        if ($lesson->status === 'completed' || $lesson->status === 'expired') {
            return response()->json(['message' => 'lesson '.$lesson->status]);
        }

        try {
            DB::beginTransaction();

            $isTeacher = $user->teacher?->id === $lesson->teacher_id;
            $presentField = $isTeacher ? 'teacher_present' : 'student_present';

            $lesson->$presentField = false;
    
            if ($lesson->status === 'in_progress' && $meeting->session_start_time) {
                $duration = now()->diffInMinutes($meeting->session_start_time);
                $meeting->total_duration += $duration;
                $meeting->session_start_time = null;
    
                if ($meeting->total_duration >= 41) {
                    $lesson->teacher_present = true;
                    $lesson->student_present = true;
                    $lesson->save();
                    $meeting->save();
                    $this->endMeeting($meeting);
                } elseif ($duration >= 15) {
                    $lesson->status = 'paused';
                    $lesson->teacher_present = true;
                    $lesson->student_present = true;
                } else {
                    $lesson->status = 'paused';
                    $lesson->$presentField  = false;
                }
            } else if ($lesson->status === 'scheduled' ) {
                $lessonStartTime = \Carbon\Carbon::parse($lesson->date . ' ' . $lesson->time); // Assuming `date` and `time` are fields on the lesson
                $currentTime = now();
                $timeDifference = $currentTime->diffInMinutes($lessonStartTime, false); // false returns a negative value if the time is in the past
                // If more than 15 minutes have passed after the scheduled time, mark lesson as completed
                if ($timeDifference <= -15) {
                    $lesson->status = 'ww';
                    $lesson->$presentField = true;
                    $lesson->save();
                    $meeting->save();
                    $this->endMeeting($meeting);
                    
                    $receiver = User::findOrFail($lesson->student->user_id);
                    $receiver->notify(new LessonNotification($lesson, 'expired', 'student', $receiver->id));
                    $receiver = User::findOrFail($lesson->teacher->user_id);
                    $receiver->notify(new LessonNotification($lesson, 'expired', 'teacher', $receiver->id));
                }
            }

            $lesson->save();
            $meeting->save();

            DB::commit();

            return response()->json([
                'lesson_status' => $lesson->status,
                'total_duration' => $meeting->total_duration,
            ]);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json(['message' => 'Error leaving meeting: '.$e->getMessage()], 500);
        }
    }

    public function endMeeting(Meeting $meeting)
    {
        $apiKey = env('WHEREBY_API_KEY');
        $client = new Client();

        try {
            $roomId = $meeting->room_id;

            $response = $client->delete("https://api.whereby.dev/v1/meetings/{$roomId}", [
                'headers' => [
                    'Authorization' => 'Bearer '.$apiKey,
                ],
            ]);

            if ($response->getStatusCode() === 204) {
                $earningDetails = (new TeacherPaymentController())->getEarningsDetails();

                // Modify teacher balance
                $teacher = $meeting->lesson->teacher;
                if($meeting->lesson->teacher_present){
                    if($meeting->lesson->student_present){
                        $teacher->balance += $earningDetails['completedLessons'];
                        $meeting->lesson->status = 'completed';
                        $meeting->lesson->save();
                    }else{
                        $teacher->balance += $earningDetails["studentAbsentLessons"];
                    }
                }else {
                    $teacher->balance += $earningDetails['teacherAbsentLessons']; // $earningDetails['teacherAbsentLessons'] is negatif
                }
                $teacher->save();

                return response()->json(['Meeting deleted successfully.', $meeting]);
            } else {
                return response()->json(["Failed to end meeting with room ID {$roomId}. Status code: {$response->getStatusCode()}"]);
            }
        } catch (\Exception $e) {
            return response()->json(["Error ending meeting with room ID {$roomId} due to duration limit: {$e->getMessage()}"]);
        }
    }

    public function destroy(Meeting $meeting)
    {
        $meeting->delete();

        return response()->json([
            'message' => 'Meeting deleted successfully!',
        ]);
    }

    public function authorizeMeetingAccess($lesson, $user)
    {
        // Ensure the user is the teacher, student, or admin
        if (!$user->hasRole('admin') && $lesson->teacher_id !== $user->teacher?->id && $lesson->student_id !== $user->student?->id) {
            return response()->json(['error' => 'Unauthorized access'], 403);
        }
    }
}
