<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\User;
use App\Models\Course;
use App\Models\Lesson;
use App\Models\Student;
use App\Models\Teacher;
use App\Events\UpdateCredit;
use Illuminate\Http\Request;
use App\Models\ReprogrammedLesson;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use App\Models\TeacherAvailability;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use App\Notifications\LessonNotification;

class LessonController extends Controller
{
    public function index()
    {
        $lessons = Lesson::select('lesson_title', 'date', 'time', 'course_id', 'status', 'lesson_type', 'id', 'teacher_id', 'student_id', 'description')
            ->with(['course:id,course_title',
             'teacher' => function ($query) {
                $query->select('id', 'first_name', 'last_name', 'user_id')->with(['user:id,timezone']);
            }, 
             'student' => function ($query) {
                $query->select('id', 'first_name', 'last_name', 'user_id')->with(['user:id,timezone']);
            }])
            ->where('status', '!=', 'pending_schedule')
            ->orderBy('updated_at', 'desc')
            ->get()
            ->map(function ($lesson) {
                return [
                    'id' => $lesson->id,
                    'lesson_title' => $lesson->lesson_title,
                    'date' => $lesson->date,
                    'time' => $lesson->time,
                    'course_id' => $lesson->course_id,
                    'course_title' => $lesson->course->course_title,
                    'status' => $lesson->status,
                    'lesson_type' => $lesson->lesson_type,
                    'description' => $lesson->description,
                    'teacher_id' => $lesson->teacher_id,
                    'teacher_name' => $lesson->teacher ? $lesson->teacher->first_name.' '.$lesson->teacher->last_name : null,
                    'student_id' => $lesson->student_id,
                    'student_name' => $lesson->student ? $lesson->student->first_name.' '.$lesson->student->last_name : null,
                    'student_timezone' => $lesson->student ? $lesson->student->user->timezone ?? 'UTC' : 'UTC',
                    'teacher_timezone' => $lesson->teacher ? $lesson->teacher->user->timezone ?? 'UTC' : 'UTC',
                ];
            });

        return response()->json(['lessons' => $lessons]);
    }

    public function store(Request $request)
    {
        try {
            DB::beginTransaction();

            $validatedData = $request->validate([
                'teacher_id' => 'nullable|exists:teachers,id',
                'student_id' => 'required|exists:students,id',
                'course_id' => 'required|exists:courses,id',
                'lesson_title' => 'required|string',
                'lesson_type' => 'required|string|in:trial,normal',
                'date' => 'required|date',
                'time' => 'required',
                'description' => 'nullable|string',
                'status' => 'required|string',
                'credit_consumed' => 'nullable|integer',
            ]);

            $validatedData['created_by'] = auth()->id();
            $student = Student::findOrFail($validatedData['student_id']);
            $teacher = Teacher::findOrFail($validatedData['teacher_id']);
            $validatedData['duration'] = $validatedData['lesson_type'] === 'trial' ? 30 : 50;
            $studentController = new StudentController();

            if($validatedData['lesson_type'] != 'trial') {
                if ($studentController->getCurrentSubscriptionstatus($student) === 'suspended') {
                    return response()->json([
                        'message' => 'The student\'s subscription is currently suspended.',
                        'message_code' => 'SUBSCRIPTION_SUSPENDED',
                    ], 400);
                }

                if ($studentController->getCredits($student) == 0) {
                    return response()->json([
                        'message' => 'The student does not have the necessary number of credits for this lesson',
                        'message_code' => 'INSUFFICIENT_CREDITS',
                    ], 400);
                }
                $studentController->deductCredits($student, 1);
                $validatedData['credit_consumed'] = 1;
            } else {
                if (!$student->free_lesson_redeemed) {
                    $student->free_lesson_redeemed = true;
                    $student->save();
                }
                $validatedData['credit_consumed'] = 0;
            }

            $lesson = Lesson::create($validatedData);

            //Notifications
            if($validatedData['status'] == 'pending'){
                $receiver = User::findOrFail($teacher->user_id);
                $receiver->notify(new LessonNotification($lesson, 'lesson_request', 'teacher', $receiver->id));
                //if admin student notification
                if (User::find(auth()->user()->id)->hasRole('admin')) {
                    $receiver = User::findOrFail($student->user_id);
                    $receiver->notify(new LessonNotification($lesson, 'lesson_request_pending', 'student', $receiver->id));
                }
            } 

            if($validatedData['status'] == 'scheduled'){
                //student notification
                $receiver = User::findOrFail($student->user_id);
                $receiver->notify(new LessonNotification($lesson, 'created', 'student', $receiver->id));

                //if admin teacher notification
                if (User::find(auth()->user()->id)->hasRole('admin')) {
                    $receiver = User::findOrFail($teacher->user_id);
                    $receiver->notify(new LessonNotification($lesson, 'created', 'teacher', $receiver->id));
                }

            }


            $lesson = Lesson::findOrFail($lesson->id);
            $lesson->load(['course', 'teacher', 'student']);

            $lessonData = [
                'id' => $lesson->id,
                'lesson_title' => $lesson->lesson_title,
                'date' => $lesson->date,
                'time' => $lesson->time,
                'course_id' => $lesson->course_id,
                'course_title' => $lesson->course->course_title,
                'status' => $lesson->status,
                'lesson_type' => $lesson->lesson_type,
                'description' => $lesson->description,
                'teacher_id' => $lesson->teacher_id,
                'teacher_name' => $lesson->teacher ? $lesson->teacher->first_name.' '.$lesson->teacher->last_name : null,
                'student_id' => $lesson->student_id,
                'student_name' => $lesson->student ? $lesson->student->first_name.' '.$lesson->student->last_name : null,
            ];
            event(new UpdateCredit($student->user->id));
            DB::commit();

            return response()->json([
                'lesson' => $lessonData,
                'message' => 'Lesson programmed successfully!',
                'message_code' => $validatedData['status'] == 'pending' ? 'LESSON_CREATE_REQUEST_SUCCESS' : 'LESSON_CREATE_SUCCESS',
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error creating lesson: ' . $e->getMessage(), [
                'file'  => $e->getFile(),
                'line'  => $e->getLine(),
                'trace' => $e->getTraceAsString(),
            ]);
        
            // Optionally log the $lesson variable without trying to convert it to a string directly.
            // You can use json_encode if the lesson is JSON-serializable.
            Log::error('Lesson data at error: ' . json_encode(JSON_PRETTY_PRINT));

            return response()->json([
                'message' => 'An error occurred while creating the lesson',
                'message_code' => 'LESSON_CREATE_ERROR',
                'error' => $e->getMessage(),
            ], 400);
        }
    }

    public function storeTrialLesson(Request $request)
    {
        try {
            DB::beginTransaction();

            // Validate the request data
            $validatedData = $request->validate([
                'student_id' => 'required|exists:students,id',
                'course_id' => 'required|exists:courses,id',
                'lesson_title' => 'required|string',
                'date' => 'required|date',
                'time' => 'required',
                'description' => 'nullable|string',
            ]);

            // Find the student and create the lesson without a teacher
            $validatedData['teacher_id'] = null; // Teacher is not assigned for trial lessons
            $validatedData['credit_consumed'] = 0; // No credits consumed for trial lessons
            $validatedData['status'] = 'pending_schedule'; // Set status to pending by default
            $validatedData['duration'] = 30;
            $validatedData['lesson_type'] = 'trial';
            $student = Student::find($validatedData['student_id']);
            // Create the lesson
            $lesson = Lesson::create($validatedData);

            if (!$student->free_lesson_redeemed) {
                $student->free_lesson_redeemed = true;
                $student->save();
            }
            // Notify the admin about the trial lesson creation
            $admins = User::role('admin')->get(); // Assuming roles are being used for admin
            foreach ($admins as $admin) {
                $admin->notify(new LessonNotification($lesson, 'trial_lesson_created', 'admin', $admin->id));
            }

            // Commit the transaction
            DB::commit();

            return response()->json([
                'lesson' => $lesson,
                'message' => 'Trial lesson created successfully!',
                'message_code' => 'TRIAL_LESSON_CREATE_SUCCESS',
            ]);
        } catch (\Exception $e) {
            // Rollback the transaction if there's an error
            DB::rollBack();

            Log::error('Error creating trial lesson: ' . $e->getMessage());

            return response()->json([
                'message' => 'An error occurred while creating the trial lesson',
                'message_code' => 'TRIAL_LESSON_CREATE_ERROR',
                'error' => $e->getMessage(),
            ], 400);
        }
    }

    public function finalisedTrialLesson(Request $request, Lesson $lesson)
    {
        $validatedData = $request->validate([
            'teacher_id' => 'required|exists:teachers,id',
            'status' => 'required|string',
            'date' => 'required|date',
            'time' => 'required',
        ]);

        DB::beginTransaction();

        try {

            $validatedData['created_by'] = auth()->id();
            $lesson->update($validatedData);

            //notification teacher
            $notificationType = $validatedData['status'] == 'scheduled' ? 'created' : 'lesson_request';
            $receiver = User::findOrFail($lesson->teacher->user_id);
            $receiver->notify(new LessonNotification($lesson, $notificationType, 'teacher', $receiver->id));

            //notification student
            $notificationType = $validatedData['status'] == 'scheduled' ? 'created' : 'lesson_request_pending';
            $receiver = User::findOrFail($lesson->student->user_id);
            $receiver->notify(new LessonNotification($lesson, $notificationType, 'student', $receiver->id));


            $lesson =  [
                'id' => $lesson->id,
                'lesson_title' => $lesson->lesson_title,
                'date' => $lesson->date,
                'time' => $lesson->time,
                'course_id' => $lesson->course_id,
                'course_title' => $lesson->course->course_title,
                'status' => $lesson->status,
                'lesson_type' => $lesson->lesson_type,
                'description' => $lesson->description,
                'teacher_id' => $lesson->teacher_id,
                'teacher_name' => $lesson->teacher ? $lesson->teacher->first_name.' '.$lesson->teacher->last_name : null,
                'student_id' => $lesson->student_id,
                'student_name' => $lesson->student ? $lesson->student->first_name.' '.$lesson->student->last_name : null,
            ];

            DB::commit();

            return response()->json([
                'lesson' => $lesson,
                'message' => 'Trial Lesson finalised updated successfully!',
                'message_code' => 'TRIAL_LESSON_FINALISED_SUCCESS',
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error finalising trial lesson: '.$e->getMessage());

            return response()->json([
                'message' => 'Error finalising trial lesson: '.$e->getMessage(),
                'message_code' => 'TRIAL_LESSON_FINALISED_ERROR',
            ], 400);
        }
    }

    public function show(Lesson $lesson)
    {
        $lesson->load([
            'course:id,course_title', 'meeting',
            'teacher' => function ($query) {
                $query->with(['user:id,timezone']);
            },
            'student' => function ($query) {
                $query->with(['user:id,timezone']);
            },
        ]);

        $lessonData = [
            'id' => $lesson->id,
            'lesson_title' => $lesson->lesson_title,
            'date' => $lesson->date,
            'time' => $lesson->time,
            'course_id' => $lesson->course_id,
            'course_title' => $lesson->course->course_title,
            'status' => $lesson->status,
            'lesson_type' => $lesson->lesson_type,
            'description' => $lesson->description,
            'teacher_present' => $lesson->teacher_present,
            'student_present' => $lesson->student_present,
            'credit_consumed' => $lesson->credit_consumed ? $lesson->credit_consumed : 0,
            'teacher_id' => $lesson->teacher_id,
            'teacher_name' => $lesson->teacher ? $lesson->teacher->first_name.' '.$lesson->teacher->last_name : null,
            'student_id' => $lesson->student_id,
            'student_name' => $lesson->student ? $lesson->student->first_name.' '.$lesson->student->last_name : null,
            'student_timezone' => $lesson->student ? $lesson->student->user->timezone ?? 'UTC' : 'UTC',
            'teacher_timezone' => $lesson->teacher ? $lesson->teacher->user->timezone ?? 'UTC' : 'UTC',
        ];

        return response()->json(['lesson' => $lessonData]);
    }

    public function reprogramLesson(Request $request, Lesson $lesson) 
    {
        // Step 1: Validate input data
        $validatedData = $request->validate([
            'new_date' => 'required|date|after_or_equal:today',
            'new_time' => 'required',
            'motif' => 'nullable|string',
        ]);

        // Check reprogramming deadline only if date and time were sent
        $reprogrammingDeadline = Carbon::parse($lesson->date.' '.$lesson->time)->subMinutes(15);
        if (now()->greaterThan($reprogrammingDeadline)) {
            return response()->json([
                'message' => 'Lesson reprogramming deadline has passed.',
                'message_code' => 'LESSON_REPROGRAMME_TIME_PASSE',
            ], 400);
        }

        // Prepare new lesson times
        $newStartTime = Carbon::parse($validatedData['new_date'] . ' ' . $validatedData['new_time']);
        $newEndTime = $newStartTime->copy()->addMinutes($lesson->duration); // Define lessonDuration
        $existingLessons = Lesson::where(function ($query) use ($lesson) {
                $query->where('teacher_id', $lesson->teacher_id)
                      ->orWhere('student_id', $lesson->student_id);
                })
                ->where('date', $validatedData['new_date'])
                ->get();
    
    
        foreach ($existingLessons as $existingLesson) {
            $existingStartTime = Carbon::parse($existingLesson->date . ' ' . $existingLesson->time);
            $existingEndTime = $existingStartTime->copy()->addMinutes($existingLesson->duration); // Same lesson duration
    
            // If this is the lesson it self, skip the overlap check
            if ($lesson->id === $existingLesson->id) {
                continue;
            }

            // Check for overlaps
            if (
                $newStartTime->between($existingStartTime->subMinutes(10), $existingEndTime->addMinutes(10)) || 
                $newEndTime->between($existingStartTime->subMinutes(10), $existingEndTime->addMinutes(10))
            ) {
                return response()->json([
                    'message' => 'Lesson cannot be created because of lesson overlapping.',
                    'message_code' => 'LESSON_REPROGRAMME_OVERLAP_ERROR',
                ], 400);
            }
        }
    
        // Step 5: If checks pass, proceed to reprogram the lesson
        DB::beginTransaction();
        
        try {
            // Create the reprogrammed lesson
            $reprogramLesson = new ReprogrammedLesson([
                'old_date' => $lesson->date,
                'old_time' => $lesson->time,
                'motif' => $validatedData['motif'] ?? null,
            ]);
    
            // Associate the reprogrammed lesson with the original lesson
            $lesson->reprogrammedLesson()->save($reprogramLesson);
            
            // Update the original lesson
            $lesson->update([
                'is_reprogrammed' => true,
                'date' => $validatedData['new_date'],
                'time' => $validatedData['new_time'],
            ]);

            if (User::find(auth()->user()->id)->hasRole('admin')) {
                $receiver = User::findOrFail($lesson->teacher->user_id);
                $receiver->notify(new LessonNotification($lesson, 'reprogrammed', 'teacher', $receiver->id));
            }

            $receiver = User::findOrFail($lesson->student->user_id);
            $receiver->notify(new LessonNotification($lesson, 'reprogrammed', 'student', $receiver->id));
            DB::commit();
            
            return response()->json([
                'message' => 'Lesson reprogrammed successfully.',
                'message_code' => 'LESSON_REPROGRAMME_SUCCESS',
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            
            return response()->json([
                'message' => 'Error during lesson reprogramming: ' . $e->getMessage(),
                'message_code' => 'LESSON_REPROGRAMME_ERROR',
            ], 400);
        }
    }
    
    public function update(Request $request, Lesson $lesson)
    {
        $validatedData = $request->validate([
            'teacher_id' => 'nullable|exists:teachers,id',
            'student_id' => 'nullable|exists:students,id',
            'course_id' => 'nullable|exists:courses,id',
            'lesson_title' => 'nullable|string',
            'lesson_type' => 'nullable|string|in:trial,normal',
            'status' => 'nullable|string',
            'date' => 'nullable|date',
            'time' => 'nullable',
            'description' => 'nullable|string',
            'credit_consumed' => 'nullable|integer',
        ]);
        

        DB::beginTransaction();

        try {
            if($lesson->lesson_type != 'trial') {
                if (isset($validatedData['student_id'])) {
                    $student = Student::find($validatedData['student_id']);
                    $studentController = new StudentController();

                    if ($studentController->getCredits($student) == 0) {
                        return response()->json([
                            'message' => 'The student does not have the necessary number of credits for this lesson',
                            'message_code' => 'INSUFFICIENT_CREDITS',
                        ], 400);
                    } else {
                        $studentController->deductCredits($student, 1);
                        $validatedData['credit_consumed'] = 1;
                    }
                }
            }

            if (isset($validatedData['date']) && isset($validatedData['time'])) {
                $newDate = $validatedData['date'] ?? null;
                $newTime = $validatedData['time'] ?? null;
                unset($validatedData['date']);
                unset($validatedData['time']);

                if ($newDate && $newTime) {
                    // Check reprogramming deadline only if date and time were sent
                    $reprogrammingDeadline = Carbon::parse($lesson->date.' '.$lesson->time)->subMinutes(15);
                    if (now()->greaterThan($reprogrammingDeadline)) {
                        return response()->json([
                            'message' => 'Lesson reprogramming deadline has passed.',
                            'message_code' => 'LESSON_REPROGRAMME_TIME_PASSE',
                        ], 400);
                    }
                    $request->request->add(['new_date' => $newDate]);
                    $request->request->add(['new_time' => $newTime]);
                    $reprogramResult = $this->reprogramLesson($request, $lesson);

                    if ($reprogramResult->getStatusCode() !== 200) {
                        throw new \Exception($reprogramResult->getData()->message, $reprogramResult->getStatusCode());
                    }
                } else {
                    throw new \Exception('Date and time are required for reprogramming', 400);
                }
            }

            $lesson->update($validatedData);
            if (User::find(auth()->user()->id)->hasRole('admin')) {
                $receiver = User::findOrFail($lesson->teacher->user_id);
                $receiver->notify(new LessonNotification($lesson, 'updated', 'teacher', $receiver->id));
            }

            $receiver = User::findOrFail($lesson->student->user_id);
            $receiver->notify(new LessonNotification($lesson, 'updated', 'student', $receiver->id));

            $lesson =  [
                'id' => $lesson->id,
                'lesson_title' => $lesson->lesson_title,
                'date' => $lesson->date,
                'time' => $lesson->time,
                'course_id' => $lesson->course_id,
                'course_title' => $lesson->course->course_title,
                'status' => $lesson->status,
                'lesson_type' => $lesson->lesson_type,
                'description' => $lesson->description,
                'teacher_id' => $lesson->teacher_id,
                'teacher_name' => $lesson->teacher ? $lesson->teacher->first_name.' '.$lesson->teacher->last_name : null,
                'student_id' => $lesson->student_id,
                'student_name' => $lesson->student ? $lesson->student->first_name.' '.$lesson->student->last_name : null,
            ];

            event(new UpdateCredit($receiver->id));
            DB::commit();
    
            return response()->json([
                'lesson' => $lesson,
                'message' => 'Lesson updated successfully!',
                'message_code' => 'LESSON_UPDATE_SUCCESS',
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error updating lesson: '.$e->getMessage());

            return response()->json([
                'message' => 'Error during lesson update : '.$e->getMessage(),
                'message_code' => 'LESSON_UPDATE_ERROR',
            ], 400);
        }
    }

    public function statusUpdate(Request $request, Lesson $lesson)
    {
        try {
            DB::beginTransaction();
            $validatedData = $request->validate([
                'status' => 'required|string',
            ]);
            $student = Student::findOrFail($lesson->student_id);
            $teacher = Teacher::findOrFail($lesson->teacher_id);
            $passedStatus = $lesson->status;
            $lesson->update($validatedData);

            if ($validatedData['status'] == 'canceled') {
                
                if($passedStatus == 'scheduled') {
                    $receiver = User::findOrFail($student->user_id);
                    $receiver->notify(new LessonNotification($lesson, 'canceled', 'student', $receiver->id));

                    //if admin teacher notification
                    if (User::find(auth()->user()->id)->hasRole('admin')) {
                        $receiver = User::findOrFail($teacher->user_id);
                        $receiver->notify(new LessonNotification($lesson, 'canceled', 'teacher', $receiver->id));
                    }
                    if ($lesson->lesson_type == 'trial' && !User::find(auth()->user()->id)->hasRole('admin')) {
                        $admins = User::role('admin')->get(); 
                        foreach ($admins as $admin) {
                            $admin->notify(new LessonNotification($lesson, 'canceled', 'admin', $receiver->id));
                        }
                        
                    }
                }

                if($passedStatus == 'pending') {
                    $receiver = User::findOrFail($student->user_id);
                    $receiver->notify(new LessonNotification($lesson, 'rejected', 'student', $receiver->id));
                    //if admin created notify them
                    $createdBy = User::find($lesson->created_by);
                    if ($createdBy->hasRole('admin') && !User::find(auth()->user()->id)->hasRole('admin')) {
                        $createdBy->notify(new LessonNotification($lesson, 'rejected', 'admin', $createdBy->id));
                    }
                }

                if ($lesson->lesson_type == 'trial' && $passedStatus == 'pending' && !User::find(auth()->user()->id)->hasRole('admin')) {

                    DB::commit();
                    return response()->json([
                        'message' => 'Lesson status updated successfully!',
                        'message_code' => 'TRIAL_LESSON_REJECT_SUCCESS',
                    ]);
                }

                if ($lesson->lesson_type == 'normal' && $lesson->credit_consumed >= 1) {
                    $currentSubscription = $student->subscriptions()
                        ->whereIn('status', ['active', 'suspended'])
                        ->orderByDesc('created_at')
                        ->first();
    
                    if ($currentSubscription) {
                        // Return credit and check subscription end date
                        $currentSubscription->credits_remaining += $lesson->credit_consumed ;
                        $currentSubscription->save();
                        $lesson->credit_consumed = 0;

                        // Notify student about credit return
                        $receiver = User::findOrFail($student->user_id);
                        $receiver->notify(new LessonNotification($lesson, 'credit_returned_active', 'student', $receiver->id));
                    } else {
                        // Handle expired subscription
                        $activatedSubscription = null;
                        $lastSubscriptions = $student->subscriptions()
                                ->whereIn('status', ['expired', 'finished', 'canceled'])
                                ->orderByDesc('created_at')
                                ->get();
    
                        foreach ($lastSubscriptions as $subscription) {
                            if ($subscription->status !== 'canceled' || ($subscription->payment && $subscription->payment->status === 'completed')) {
                                $subscription->update([
                                    'status' => 'active',
                                    'end_date' => Carbon::now()->addDays(14),
                                    'credits_remaining' => $subscription->credits_remaining + $lesson->credit_consumed ,
                                ]);
                                $activatedSubscription = $subscription;
                                $lesson->credit_consumed = 0;

                                // Notify student about credit return
                                $receiver = User::findOrFail($student->user_id);
                                $receiver->notify(new LessonNotification($lesson, 'credit_returned_passed', 'student', $receiver->id));
                                break;
                            }
                        }

                        // If no subscription was activated
                        if (!$activatedSubscription) {
                            Log::error('No suitable subscription found for student when returning credit.', [
                                'student_id' => $student->id
                            ]);

                            // Notify the admin about the situation
                            $admins = User::role('admin')->get();
                            foreach ($admins as $admin) {
                                $admin->notify(new LessonNotification($lesson, 'credit_not_return', 'admin', $admin->id));
                            }

                            // Notify the student about the situation
                            $receiver = User::findOrFail($student->user_id);
                            $receiver->notify(new LessonNotification($lesson, 'credit_not_return', 'student', $receiver->id));
                        }
                    }
    
                    $lesson->save();
                }
            }

            if ($validatedData['status'] == 'scheduled') {
                $receiver = User::findOrFail($student->user_id);
                $receiver->notify(new LessonNotification($lesson, 'accepted', 'student', $receiver->id));

                //if admin teacher notification
                if (User::find(auth()->user()->id)->hasRole('admin')) {
                    $receiver = User::findOrFail($teacher->user_id);
                    $receiver->notify(new LessonNotification($lesson, 'accepted', 'teacher', $receiver->id));
                }
            }

            event(new UpdateCredit($student->user->id));
            DB::commit();
            return response()->json([
                'message' => 'Lesson status updated successfully!',
                'message_code' => 'LESSON_STATUS_'.strtoupper($validatedData['status']),
            ]);
        } catch (\Exception $e) {
            DB::rollBack();

            Log::error('Error creating lesson: '.$e->getMessage());

            return response()->json([
                'message' => 'An error occurred while lesson status update',
                'message_code' => 'LESSON_UPDATE_ERROR',
                'error' => $e->getMessage(),
            ], 400);
        }
    }

    public function trialLessonList()
    {
        $lessons = Lesson::where('status', 'pending_schedule')
                           ->select('date', 'time', 'course_id', 'status', 'id', 'student_id','lesson_title')
                           ->orderBy('updated_at', 'desc')
                           ->get()
                           ->map(function ($lesson) {
                            return [
                                'id' => $lesson->id,
                                'lesson_title' => $lesson->lesson_title,
                                'date' => $lesson->date,
                                'time' => $lesson->time,
                                'course_id' => $lesson->course_id,
                                'course_title' => $lesson->course->course_title,
                                'status' => $lesson->status,
                                'student_id' => $lesson->student_id,
                                'student' => [
                                    'name' =>$lesson->student ? $lesson->student->first_name.' '.$lesson->student->last_name : null,
                                    'phone_number' =>$lesson->student ? $lesson->student->phone_number : null,
                                    'class_name' =>$lesson->student ? $lesson->student->academicLevelClass->class_name : null,
                                ],
                            ];
                        });

        return response()->json(['lessons' => $lessons]);
    }

    public function destroy(Lesson $lesson)
    {
        $lesson->delete();

        return response()->json([
            'message' => 'Lesson deleted successfully!',
            'message_code' => 'LESSON_DELETE_SUCCESS',
        ]);
    }

    public function getStatus(Lesson $lesson)
    {
        return response()->json(['status' => $lesson->status]);
    }

    public function getAvailabilities(Request $request)
    {   
        $validatedData = $request->validate([
            'teacher_id' => 'required|exists:teachers,id',
            'student_id' => 'required|exists:students,id',
        ]);

        $currentTime = Carbon::now();
        $userTimezone = auth()->user()->timezone ?? config('app.timezone');
        // Get all future availabilities for the teacher
        $availabilities = TeacherAvailability::where('teacher_id', $validatedData['teacher_id'])
            ->where('date', '>=', $currentTime->toDateString()) // Fetch future dates only
            ->get();

        $availabilityMarkers = $availabilities->map(function ($availability) use ($validatedData, $userTimezone) {
            $availabilityStart = Carbon::parse($availability->start_time);
            $availabilityEnd = Carbon::parse($availability->end_time);

            $allLessons = Lesson::where(function ($query) use ($validatedData) {
                $query->where('teacher_id', $validatedData['teacher_id'])
                      ->orWhere('student_id', $validatedData['student_id']);
                })
                ->where('date', $availability->date)
                ->get()  // Execute the query and get the results
                ->unique('id')
                ->map(function ($lesson) {
                    $lessonStartTime = Carbon::parse($lesson->time);
                    $lessonEndTime = $lessonStartTime->copy()->addMinutes(($lesson->duration)); // Assuming lesson duration is 1 hour
                    return ['start' => $lessonStartTime->subMinutes(10), 'end' => $lessonEndTime->addMinutes(10)];
                })
                ->sortBy('start') // Sort by lesson start time
                ->values(); 

            $availableSlots = $this->calculateAvailableTimeSlots($availabilityStart, $availabilityEnd, $allLessons);            

            // After deducting lessons, if the remaining available time is less than 30 minutes, consider it unavailable
            if (empty($availableSlots)) {
                return [
                    'date' => $availability->date,
                    'type' => 'dot',
                    'color' => 'red', // Fully booked
                    'tooltip' => [
                        'text' => 'Fully booked',
                        'color' => 'red',
                    ],
                ];    
            }

              // Convert available slots to user's timezone
            $slotsInUserTimezone = array_map(function($slot) use ($userTimezone) {
                $start = Carbon::parse($slot['start'])->setTimezone($userTimezone);
                $end = Carbon::parse($slot['end'])->setTimezone($userTimezone);
                return [
                    'text' => $start->format('H:i') . ' - ' . $end->format('H:i'),
                    'color' => 'green',
                ];
            }, $availableSlots);

            // Return availability marker with tooltips in user's timezone
            return [
                'date' => $availability->date,
                'type' => 'dot',
                'color' => 'green', // Assume availability for now
                'tooltip' => $slotsInUserTimezone,
            ];
        })->filter();

        return response()->json(['availabilityMarkers' => $availabilityMarkers]);
    }

    function calculateAvailableTimeSlots($availabilityStart, $availabilityEnd, $allLessons)
    {
        // Convert the availability start and end to Carbon instances
        $availabilityStart = Carbon::parse($availabilityStart);
        $availabilityEnd = Carbon::parse($availabilityEnd);

        // Initialize available time slots
        $availableSlots = [];
        
        // Track the current time we're looking at, starting from the availability start
        $currentTime = $availabilityStart;

        foreach ($allLessons as $lesson) {
            $lessonStartTime = $lesson['start'];
            $lessonEndTime = $lesson['end'];

            // If the lesson starts after the current time, we have a potential free slot
            if ($currentTime->lessThan($lessonStartTime)) {
                // Calculate the duration of the free slot
                $freeSlotMinutes = $currentTime->diffInMinutes($lessonStartTime);

                // Only consider free slots that are 30 minutes or more
                if ($freeSlotMinutes >= 30) {
                    $availableSlots[] = [
                        'start' => $currentTime->format('H:i'),
                        'end' => $lessonStartTime->format('H:i'),
                    ];
                }
            }

                $currentTime = $lessonEndTime;
        }

        // If there's time left after the last lesson, it's also an available slot
        if ($currentTime->lessThan($availabilityEnd)) {
            $freeSlotMinutes = $currentTime->diffInMinutes($availabilityEnd);
            
            // Only consider free slots that are 30 minutes or more
            if ($freeSlotMinutes >= 30) {
                $availableSlots[] = [
                    'start' => $currentTime->format('H:i'),
                    'end' => $availabilityEnd->format('H:i'),
                ];
            }
        }

        return $availableSlots;
    }

    public function getLessonTypes()
    {
        $lessonTypes = Lesson::select('lesson_type')->distinct()->get()->pluck('lesson_type')->toArray();
        return response()->json(['lesson_types' => $lessonTypes]);
    }

}
