<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use App\Services\GoogleDriveService;
use App\Models\File;
use App\Models\User;

class GoogleDriveController extends Controller
{
    public function __construct(protected GoogleDriveService $driveService) {}

    /**
     * List files and folders
     */
    public function index(Request $request)
    {
        $folderId = $request->query('folderId');
        $files = $this->driveService->listFiles($folderId);

        $authUser = User::find(auth()->user()->id);
        $authUserId = $authUser->id;
        $teacherCourses = null;
        $studentCourses = null;

        if ($folderId === 'root') {
            if ($authUser->hasRole('teacher')) {
                // Get courses linked to the teacher from the teacher_course table
                $teacherCourses = DB::table('teacher_courses')
                    ->where('teacher_id', $authUser->teacher->id)
                    ->pluck('course_id');
            } elseif ($authUser->hasRole('student')) {
                // Get courses linked to the student from the student_courses table
                $studentCourses = DB::table('student_courses')
                    ->where('student_id', $authUser->student->id)
                    ->pluck('course_id');
            }
        }

        // Add created_by and can_edit information for each file
        $filesWithDetails = collect($files)->map(function ($file) use ($authUser, $authUserId, $teacherCourses, $studentCourses) {
            $fileRecord = DB::table('files')->where('file_drive_id', $file['id'])->first();
            if ($fileRecord) {
                $createdById = $fileRecord->created_by;
                $user = User::find($createdById);

                if ($user) {
                    if ($authUser->hasRole('admin')) {
                        $file['created_by'] = $user->hasRole('admin') ? 'Admin' : ($user->hasRole('teacher') ? 'Tuteur' : null);
                    } elseif ($authUser->hasRole('teacher')) {
                        $file['created_by'] = $createdById === $authUserId ? 'Moi' : 'Autre';
                    }else {
                        $file['created_by'] = null;
                    }
                } else {
                    $file['created_by'] = null;
                }

                // Determine if the authenticated user can edit the file
                $file['can_edit'] = $authUser->hasRole('admin') || $createdById === $authUserId;
                $file['is_folder'] = $fileRecord->is_folder;

                // Filter files based on teacher's courses if applicable
                if ($authUser->hasRole('teacher') && isset($teacherCourses) && !$teacherCourses->contains($fileRecord->course_id)) {
                    return null; // Exclude files not linked to the teacher's courses
                }

                // Filter files based on student's courses if applicable
                if ($authUser->hasRole('student') && isset($studentCourses) && !$studentCourses->contains($fileRecord->course_id)) {
                    return null; // Exclude files not linked to the student's courses
                }
            } else {
                $file['created_by'] = null;
                $file['is_folder'] = false;
                $file['can_edit'] = $authUser->hasRole('admin');
            }

            return $file;
        })->filter()->values(); // Remove null entries and reset keys

        return response()->json([
            'files' => $filesWithDetails->isEmpty() ? [] : $filesWithDetails->toArray(),
            'message' => 'Files retrieved successfully',
            'current_folder_id' => $folderId
        ]);
    }

    /**
     * Create a new folder
     */
    public function createFolder(Request $request)
    {
        $validatedData = $request->validate([
            'name' => 'required|string|max:255',
            'parent_id' => 'nullable|string',
            'course_id' => 'nullable|integer',
        ]);

        $folderId = $this->driveService->createFolder(
            $validatedData['name'], 
            $validatedData['parent_id'] ?? 'root'
        );

        File::create([
            'file_drive_id' => $folderId,
            'created_by' => auth()->user()->id,
            'is_folder' => isset($validatedData['course_id']) ? true : false,
            'course_id' => $validatedData['course_id'] ?? null,
        ]);

        return response()->json([
            'success' => true,
            'folder_id' => $folderId,
            'message' => 'Folder created successfully'
        ]);
    }

    /**
     * Upload a file
     */
    public function uploadFiles(Request $request)
    {
        $validatedData = $request->validate([
            'files.*' => 'required|file|max:104857600', // 100MB limit per file
            'folder_id' => 'nullable|string',
        ]);

        $uploadedFiles = [];
        foreach ($validatedData['files'] as $file) {
            $fileInfo = $this->driveService->uploadFile(
                $file,
                $validatedData['folder_id']
            );
            $uploadedFiles[] = $fileInfo;

            File::create([
                'file_drive_id' => $fileInfo['id'],
                'created_by' => auth()->user()->id,
                'course_id' => 1,
            ]);
        }

        return response()->json([
            'files' => $uploadedFiles,
            'message' => 'Files uploaded successfully'
        ]);
    }

    /**
     * Delete a file or folder
     */
    public function destroy($fileId)
    {
        $result = $this->driveService->delete($fileId);
        
        if ($result) {
            $file = File::where('file_drive_id', $fileId)->first();
            if ($file) {
                $file->delete(); 
            }

            return response()->json([
                'success' => true,
                'message' => 'Item deleted successfully'
            ]);
        }
        
        return response()->json([
            'success' => false,
            'message' => 'Failed to delete item'
        ], 400);
    }
    
    /**
     * Get file details for preview
     */
    public function getFileDetails($fileId)
    {
        $fileInfo = $this->driveService->getFile($fileId);
        
        if (!$fileInfo) {
            return response()->json([
                'success' => false,
                'message' => 'File not found'
            ], 404);
        }
        
        return response()->json([
            'success' => true,
            'file' => $fileInfo
        ]);
    }

    /**
     * Get all folders and subfolders
     */
    public function getAllFolders(Request $request)
    {
        $folderId = $request->query('folderId', null);
        $folders = $this->driveService->getAllFolders($folderId);

        return response()->json([
            'success' => true,
            'folders' => $folders,
            'message' => 'Folders retrieved successfully'
        ]);
    }

    /**
     * Rename a folder or a file
     */
    public function renameFile(Request $request, $fileId)
    {
        $validatedData = $request->validate([
            'new_name' => 'required|string|max:255'
        ]);

        $updatedFile = $this->driveService->renameFile($fileId, $validatedData['new_name']);

        return response()->json([
            'message' => 'File renamed successfully',
            'file' => $updatedFile
        ]);
    }

    /**
     * Move a folder or a file
     */

    public function moveFile(Request $request, $fileId)
    {
        $validatedData = $request->validate([
            'destination_folder_id' => 'required|string'
        ]);

        $updatedFile = $this->driveService->moveFile($fileId, $validatedData['destination_folder_id']);

        return response()->json([
            'message' => 'File moved successfully',
            'file' => $updatedFile
        ]);
    }

}