<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Course;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;

class CourseController extends Controller
{
    public function index()
    {
        $courses = Course::with('courseCategory', 'teachers', 'students')->orderBy('updated_at', 'desc')->get()
                    ->map(function ($course) {
                        return [
                            'id' => $course->id,
                            'course_title' => $course->course_title,
                            'course_description' => $course->course_description,
                            'course_category_name' => $course->courseCategory ? $course->courseCategory->course_category_name : null,
                            'course_category_id' => $course->courseCategory ? $course->courseCategory->id : null,
                            'teachers_count' => $course->teachers->count(),
                            'students_count' => $course->students->count(),
                            'status' => $course->status,
                            'created_at' => $course->created_at ? $course->created_at : null,
                        ];
                    });

        return response()->json(['courses' => $courses]);
    }

    public function show(Course $course)
    {
        $course->load('courseCategory');

        $courseData = [
            'id' => $course->id,
            'course_title' => $course->course_title,
            'course_description' => $course->course_description,
            'course_category_name' => $course->courseCategory ? $course->courseCategory->course_category_name : null,
            'course_category_id' => $course->courseCategory ? $course->courseCategory->id : null,
            'created_at' => $course->created_at,
            'status' => $course->status,
        ];

        return response()->json(['course' => $courseData]);
    }

    public function showTeacherStudent(Course $course)
    {
        $course->load([
            'teachers' => function ($query) {
                $query->select('id', 'user_id', 'first_name', 'last_name', 'phone_number', 'status', 'profile_image','email');
            },
            'students' => function ($query) {
                $query->select('id', 'user_id', 'first_name', 'last_name', 'phone_number', 'status', 'profile_image', 'email');
            },
        ]);

        $teachers = $course->teachers->map(function ($teacher) {
            return [
                'id' => $teacher->user_id,
                'role_id' => $teacher->id,
                'name' => $teacher->first_name.' '.$teacher->last_name,
                'phone_number' => $teacher->phone_number,
                'profile_image' => $teacher->profile_image ? asset('storage/'.$teacher->profile_image) : null,
                'status' => $teacher->status,
                'email' => $teacher->email,
                'role' => 'teacher',
            ];
        })->values();

        $students = $course->students->map(function ($student) {
            return [
                'id' => $student->user_id,
                'role_id' => $student->id,
                'name' => $student->first_name.' '.$student->last_name,
                'phone_number' => $student->phone_number,
                'profile_image' => $student->profile_image ? asset('storage/'.$student->profile_image) : null,
                'status' => $student->status,
                'email' => $student->email,
                'role' => 'student',
            ];
        })->values();

        $users = collect($teachers)->merge($students);

        return response()->json([
            'users' => $users,
        ]);
    }

    public function showStudents()
    {
        $courses = Course::with(['students:id,first_name,last_name,user_id'])
            ->select('id', 'course_title')
            ->get()
            ->map(function ($course) {
                return [
                    'id' => $course->id,
                    'course_title' => $course->course_title,
                    'students' => $course->students->unique()->map(function ($student) {
                        return [
                            'id' => $student->id,
                            'name' => $student->first_name.' '.$student->last_name,
                            'timezone' => $student->user->timezone,
                        ];
                    }),
                ];
            });

        return response()->json([
            'coursesStudents' => $courses,
        ]);
    }

    public function showTeachers()
    {
        $courses = Course::with(['teachers:id,first_name,last_name,user_id'])
            ->select('id', 'course_title')
            ->get()
            ->map(function ($course) {
                return [
                    'id' => $course->id,
                    'course_title' => $course->course_title,
                    'teachers' => $course->teachers->unique()->map(function ($teacher) {
                        return [
                            'id' => $teacher->id,
                            'name' => $teacher->first_name.' '.$teacher->last_name,
                            'timezone' => $teacher->user->timezone,
                        ];
                    }),
                ];
            });

        return response()->json([
            'coursesStudents' => $courses,
        ]);
    }

    public function store(Request $request)
    {
        $validatedData = $request->validate([
            'course_title' => 'required|string',
            'course_description' => 'nullable|string',
            'course_category_id' => 'required|exists:course_categories,id',
        ]);

        $course = Course::create($validatedData);
        $request = new \Illuminate\Http\Request();
        $request->replace([
            'name' => $course->course_title,
            'parent_id' => 'root',
            'course_id' => $course->id,
        ]);
        
        app('App\Http\Controllers\GoogleDriveController')->createFolder($request);
        $course->load(['courseCategory']);
        $course->save();

        $courseData = [
            'id' => $course->id,
            'course_title' => $course->course_title,
            'course_description' => $course->course_description,
            'course_category_name' => $course->courseCategory ? $course->courseCategory->course_category_name : null,
            'course_category_id' => $course->courseCategory ? $course->courseCategory->id : null,
            'teachers_count' => 0, // New course, so no teachers yet
            'students_count' => 0, // New course, so no students yet
            'created_at' => $course->created_at,
            'status' => $course->status, // This will be 'active' by default
        ];

        return response()->json([
            'course' => $courseData,
            'message' => 'Course created successfully!',
            'message_code' => 'COURSE_CREATE_SUCCESS',
        ]);
    }

    public function update(Request $request, Course $course)
    {
        $validatedData = $request->validate([
            'course_title' => 'nullable|string',
            'course_description' => 'nullable|string',
            'course_category_id' => 'nullable|exists:course_categories,id',
        ]);

        if($validatedData['course_title']){
            $request = new \Illuminate\Http\Request();
            $request->replace([
                'new_name' => $validatedData['course_title']
            ]);
            $folderId = $course->folder?->file_drive_id;
            if ($folderId) {
                app('App\Http\Controllers\GoogleDriveController')->renameFile($request, $folderId);
            }
        }

        $course->update($validatedData);

        return response()->json([
            'message' => 'Course updated successfully!',
            'message_code' => 'COURSE_UPDATE_SUCCESS',
        ]);
    }

    public function statusUpdate(Request $request, Course $course)
    {
        $validatedData = $request->validate([
            'status' => 'required|string',
        ]);

        $course->update($validatedData);

        return response()->json([
            'message' => 'Course status updated successfully!',
            'message_code' => 'COURSE_STATUS_'.strtoupper($validatedData['status']),
        ]);
    }

    public function destroy(Course $course)
    {
        $folderId = $course->folder?->file_drive_id;
        if ($folderId) {
            app('App\Http\Controllers\GoogleDriveController')->destroy($folderId);
        }

        $course->delete();

        return response()->json([
            'message' => 'Course deleted successfully!',
            'message_code' => 'COURSE_DELETE_SUCCESS',
        ]);
    }

    public function courseUserLink(Request $request, Course $course)
    {
        $validatedData = $request->validate([
            'users' => 'required|array', // Array of user IDs
            'users.*' => 'required|exists:users,id', // Validate user IDs
        ]);

        $userIds = $validatedData['users'];

        // Fetch users and group them by role
        $users = User::whereIn('id', $userIds)->get();
        $students = $users->filter(function ($user) {
            return $user->roles->contains('name', 'student');
        })->pluck('student.id')->toArray();

        $teachers = $users->filter(function ($user) {
            return $user->roles->contains('name', 'teacher');
        })->pluck('teacher.id')->toArray();

        // Check if students are already linked
        $existingStudentLinks = $course->students()->whereIn('student_id', $students)->get();
        $existingTeacherLinks = $course->teachers()->whereIn('teacher_id', $teachers)->get();
        if ($existingStudentLinks->isNotEmpty() || $existingTeacherLinks->isNotEmpty()) {
            return response()->json([
                'message' => 'One or more users are already associated with this course.',
                'message_code' => 'COURSE_USER_LINK_EXIST',
            ], 400);
        }

        // Link students and teachers to the course
        if (!empty($students)) {
            $course->students()->attach($students);
        }
        if (!empty($teachers)) {
            $course->teachers()->attach($teachers);
        }

        return response()->json([
            'message' => 'Users added to the course successfully.',
            'message_code' => 'COURSE_USER_LINK_SUCCESS',
        ]);
    }

    public function courseUserUnlink(Request $request, Course $course)
    {
        $validatedData = $request->validate([
            'users' => 'required|array', // Array of user IDs
            'users.*' => 'required|exists:users,id', // Validate user IDs
        ]);

        $userIds = $validatedData['users'];

        // Fetch users and group them by role
        $users = User::whereIn('id', $userIds)->get();
        $students = $users->filter(function ($user) {
            return $user->roles->contains('name', 'student');
        })->pluck('student.id')->toArray();

        $teachers = $users->filter(function ($user) {
            return $user->roles->contains('name', 'teacher');
        })->pluck('teacher.id')->toArray();

        // Check if students are linked
        $existingStudentLinks = $course->students()->whereIn('student_id', $students)->get();
        $existingTeacherLinks = $course->teachers()->whereIn('teacher_id', $teachers)->get();
        if (($existingStudentLinks->isEmpty() && !empty($students)) || ($existingTeacherLinks->isEmpty() && !empty($teachers))) {
            return response()->json([
                'message' => 'One or more users are not linked to this course.',
                'message_code' => 'COURSE_USER_LINK_NOT_EXIST',
            ], 400);
        }

        // Unlink students and teachers from the course
        if (!empty($students)) {
            $course->students()->detach($students);
        }
        if (!empty($teachers)) {
            $course->teachers()->detach($teachers);
        }

        return response()->json([
            'message' => 'Users unlinked from the course successfully.',
            'message_code' => 'COURSE_USER_UNLINK_SUCCESS',
        ]);
    }
}
