<?php

namespace App\Http\Controllers;

use App\Models\Lesson;
use App\Models\Payment;
use App\Models\Student;
use App\Models\Teacher;
use Carbon\Carbon;
use Illuminate\Routing\Controller;
use Illuminate\Support\Arr;
use Stripe\Charge;
use Stripe\Stripe;

class AdminController extends Controller
{
    public function getnSpecificAppStatistics()
    {
        // Get the total number of students
        $totalStudents = Student::count();

        // Get the total number of teachers
        $totalTeachers = Teacher::count();

        // Get the total number of completed lessons
        $totalLessonsCompleted = Lesson::where('status', 'completed')->count();

        Stripe::setApiKey(env('STRIPE_SECRET')); // Use env to get the Stripe secret key


        $totalRevenue = (int) Payment::all()->reduce(
            initial: 0,
            callback: fn(int $carry, Payment $payment) => $carry + $payment->amount
        );

        // Convert amount from cents to dollars (assuming currency is USD)
        $totalRevenue /= 100;

        $weeklyRange = [now()->subDays(7), now()];
        $monthlyRange = [now()->subDays(30), now()];
        $halfYearlyRange = [now()->subMonths(6), now()];
        $yearlyRange = [now()->subMonths(12), now()];

        // Get statistics for the specified periods
        $weeklyStats = $this->getSubscriptionAndRevenueStatsForPeriod($weeklyRange, 'daily');
        $monthlyStats = $this->getSubscriptionAndRevenueStatsForPeriod($monthlyRange, 'daily');
        $halfYearlyStats = $this->getSubscriptionAndRevenueStatsForPeriod($halfYearlyRange, 'monthly');
        $yearlyStats = $this->getSubscriptionAndRevenueStatsForPeriod($yearlyRange, 'monthly');

        // Return the results as a JSON response
        return response()->json([
            'total_students' => $totalStudents,
            'total_teachers' => $totalTeachers,
            'total_lessons_completed' => $totalLessonsCompleted,
            'total_revenue' => $totalRevenue,
            'weekly_stats' => $weeklyStats,
            'monthly_stats' => $monthlyStats,
            'half_yearly_stats' => $halfYearlyStats,
            'yearly_stats' => $yearlyStats,
        ]);
    }

    private function getSubscriptionAndRevenueStatsForPeriod($range, $groupBy)
    {
        Stripe::setApiKey(env('STRIPE_SECRET'));

        $startDate = Carbon::parse($range[0]);
        $endDate = Carbon::parse($range[1]);

        // Initialize the period with zero values
        $grouped = [];
        while ($startDate->lessThanOrEqualTo($endDate)) {
            $key = $groupBy === 'daily' ? $startDate->format('Y-m-d') : $startDate->format('Y-m');
            $grouped[$key] = [
                'actual_revenue' => 0,
                'expected_revenue' => 100,
            ];
            $groupBy === 'daily' ? $startDate->addDay() : $startDate->addMonth();
        }

        // Get charges within the period
        $charges = Charge::all(['created' => [
            'gte' => $range[0]->timestamp,
            'lte' => $range[1]->timestamp,
        ]]);

        // Sum payments within the period
        foreach ($charges->data as $charge) {
            $chargeDate = Carbon::createFromTimestamp($charge->created);
            $key = $groupBy === 'daily' ? $chargeDate->format('Y-m-d') : $chargeDate->format('Y-m');
            if (isset($grouped[$key])) {
                $grouped[$key]['actual_revenue'] += $charge->amount / 100; // Convert from cents to dollars
            }
        }

        // Convert keys to desired format and amounts to 2 decimal places
        $formattedGrouped = [];
        foreach ($grouped as $key => $values) {
            $date = Carbon::createFromFormat($groupBy === 'daily' ? 'Y-m-d' : 'Y-m', $key);
            $formattedKey = $groupBy === 'daily' ? mb_convert_encoding(ucfirst($date->isoFormat('DD MMM')), 'UTF-8', 'UTF-8') : mb_convert_encoding(ucfirst($date->isoFormat('MMM YYYY')), 'UTF-8', 'UTF-8');
            $formattedGrouped[$formattedKey] = [
                'actual_revenue' => round($values['actual_revenue'], 2),
                'expected_revenue' => round($values['expected_revenue'], 2),
            ];
        }

        return $formattedGrouped;
    }
}
