<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\DB;
use App\Models\AccountPauseHistory;
use App\Notifications\UserNotification;

class AccountPauseHistoryController extends Controller
{
    public function deactivate(Request $request, User $user)
    {
        $validatedData = $request->validate([
            'status' => 'string|required',
            'reason' => 'string|required',
        ]);

        $reason = $validatedData['reason'];
        $response = null;

        try {
            DB::beginTransaction();

            if ($user->hasRole('teacher')) {
                $response = (new TeacherController())->teacherStatusUpdate($request, $user->teacher);
                $userRole = 'teacher';
                $userName = $user->teacher->first_name.' '.$user->teacher->last_name;
                $roleId = $user->teacher->id;
            } elseif ($user->hasRole('student')) {
                $response = (new StudentController())->statusUpdate($request, $user->student);
                $userRole = 'student';
                $userName = $user->student->first_name.' '.$user->student->last_name;
                $roleId = $user->student->id;
            } else {
                return response()->json([
                    'roles' => $user,
                    'error' => 'This user have no role.',
                ], 422);
            }

            AccountPauseHistory::create([
                'user_id' => $user->id,
                'reason' => $reason,
                'deactivation_date' => Carbon::now('UTC')->toDateTimeString(),
            ]);

            $admins = User::role('admin')->get(); 
            foreach ($admins as $admin) {
                $admin->notify(new UserNotification($roleId, $userName, 'deactivated_account', $userRole, $reason));
            }

            DB::commit();
            return $response;
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'error' => 'An error occurred while deactivating the account.',
                'message' => $e->getMessage(),
            ], 500);
        }
    }

    public function activate(Request $request, User $user)
    {
        $request->validate([
            'status' => 'string|required',
        ]);

        $response = null;

        try {
            DB::beginTransaction();

            if ($user->hasRole('teacher')) {
                $response = (new TeacherController())->teacherStatusUpdate($request, $user->teacher);
                $userRole = 'teacher';
                $userName = $user->teacher->first_name.' '.$user->teacher->last_name;
                $roleId = $user->teacher->id;
            } elseif ($user->hasRole('student')) {
                $response = (new StudentController())->statusUpdate($request, $user->student);
                $userRole = 'student';
                $userName = $user->student->first_name.' '.$user->student->last_name;
                $roleId = $user->student->id;
            } else {
                return response()->json([
                    'roles' => $user,
                    'error' => 'This user have no role.',
                ], 422);
            }

            $pauseHistory = $user->accountPauseHistories()->whereNull('reactivation_date')->first();
            $pauseHistory->reactivation_date = Carbon::now('UTC')->toDateTimeString();
            $pauseHistory->save();

            $admins = User::role('admin')->get(); 
            foreach ($admins as $admin) {
                $admin->notify(new UserNotification($roleId, $userName, 'reactivated_account', $userRole, null));
            }

            DB::commit();
            return $response;
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'error' => 'An error occurred while reactivating the account.',
                'message' => $e->getMessage(),
            ], 500);
        }
    }
}
