<?php

namespace App\Console\Commands;

use Carbon\Carbon;
use Stripe\Stripe;
use App\Models\Student;
use App\Events\UpdateCredit;
use App\Models\Subscription;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\PaymentController;
use App\Http\Controllers\StudentController;
use App\Notifications\SubscriptionNotification;

class CheckSubscriptionStatus extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'subscriptions:check-status';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Check and update subscriptions statuses, handle auto-renewal, and send notifications';

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        try {
            DB::beginTransaction();

            $today = Carbon::now();
            $nextWeek = $today->copy()->addWeek();

            // 1. Handle Expired Subscriptions
            $this->handleExpiredSubscriptions($today);

            // 2. Handle About to Expire Subscriptions (send warning notifications)
            $this->handleExpiringSubscriptions($nextWeek);

            // 3. Process Auto-Renewal Subscriptions
            $this->processAutoRenewalSubscriptions($today);

            DB::commit();

            $this->info('Comprehensive subscription check completed successfully.');
            return Command::SUCCESS;

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Subscription status check failed: ' . $e->getMessage());
            $this->error('Subscription status check failed: ' . $e->getMessage());
            return Command::FAILURE;
        }
    }

    protected function handleExpiredSubscriptions(Carbon $today)
    {
        $expiredSubscriptions = Subscription::where('status', 'active')
            ->whereDate('end_date', '<', $today)
            ->get();

        foreach ($expiredSubscriptions as $subscription) {
            // Mark as expired and reset credits
            if ($subscription->auto_renew) {
                continue; // Will be handled by processAutoRenewalSubscriptions
            }

            $subscription->update([
                'status' => 'expired',
                'credits_remaining' => 0
            ]);
            $student = Student::find($subscription->student_id);
            // Notify student about expired subscription
            $student->user->notify(new SubscriptionNotification($subscription, 'subscription_expired', $student->user->id));
            event(new UpdateCredit($student->user->id));

            $this->info("Subscription ID {$subscription->id} has been marked as expired.");
        }
    }

    protected function handleExpiringSubscriptions(Carbon $nextWeek)
    {
        $expiringSubscriptions = Subscription::where('status', 'active')
            ->whereDate('end_date', '<=', $nextWeek)
            ->get();

        foreach ($expiringSubscriptions as $subscription) {
            // Check if auto-renewal is enabled
            if ($subscription->auto_renew) {
                continue; // Will be handled by processAutoRenewalSubscriptions
            }

            $student = Student::find($subscription->student_id);
            
            // Notify student about subscription expiring soon
            $student->user->notify(new SubscriptionNotification($subscription, 'subscription_expiring_soon', $student->user->id));
            
            $this->info("Sent expiration warning for Subscription ID {$subscription->id}.");
        }
    }

    protected function processAutoRenewalSubscriptions(Carbon $today)
    {
        $autoRenewSubscriptions = Subscription::where('status', 'active')
            ->where('auto_renew', true)
            ->whereDate('end_date', '<=', $today)
            ->get();

        foreach ($autoRenewSubscriptions as $subscription) {
            try {
                // Retrieve associated student and subscription plan
                $student = Student::find($subscription->student_id);
                $subscriptionPlan = $subscription->subscriptionPlan;
                $user = $student->user;
                $paymentMethods = $user->paymentMethods()->where('status', 'active')->get();
                $defaultPaymentMethodId = null;

                Stripe::setApiKey(env('STRIPE_SECRET'));
                $stripeCustomer = \Stripe\Customer::retrieve($user->stripe_customer_id);
                $defaultPaymentMethodId = $stripeCustomer->invoice_settings->default_payment_method;

                if($defaultPaymentMethodId){
                    $paymentMethods = $paymentMethods->sortBy(function ($method) use ($defaultPaymentMethodId) {
                        return $method->stripe_payment_method_id === $defaultPaymentMethodId ? 0 : 1;
                    });
                }

                foreach ($paymentMethods as $paymentMethod) {
                    try {
                        $paymentData = [
                            'payment_method_id' => $paymentMethod->stripe_payment_method_id, 
                            'amount' => $subscriptionPlan->price * 100,
                            'student_id' => $student->id,
                            'subscription_plan_id' => $subscriptionPlan->id,
                            'free_lesson' => false
                        ];
                
                        $request = new \Illuminate\Http\Request();
                        $request->replace($paymentData);
                
                        $paymentController = app(PaymentController::class);
                        $response = $paymentController->SaveCardPayment($request);
                        $responseData = $response->getData();
                
                        if ($responseData->status === 'success') {
                            $user->notify(new SubscriptionNotification($responseData->subscription, 'subscription_auto_renewed', $user->id));
                            event(new UpdateCredit($user->id));
                            $this->inFo("Successfully auto-renewed Subscription ID {$subscription->id}.");
                            break; // Exit the loop on successful renewal
                        } elseif ($responseData->status === 'processing') {
                            $this->info("Auto-renewal processing for Subscription ID {$subscription->id}.");
                            break; // Consider processing as successful
                        }
                
                        // If not successful, the loop continues to next payment method
                        $this->info("Auto-renewal failed with payment method {$paymentMethod->id}. Trying next method.");
                    } catch (\Exception $e) {
                        $this->info("Error processing payment method {$paymentMethod->id}: " . $e->getMessage());
                        // Continue to next payment method on exception
                        continue;
                    }
                }

            } catch (\Exception $e) {
                Log::error("Auto-renewal error for Subscription ID {$subscription->id}: " . $e->getMessage());
                
                // Notify about auto-renewal failure
                $user->notify(new SubscriptionNotification($subscription, 'subscription_auto_renewal_failed', $user->id));
            }
        }
    }
}
