<?php

namespace App\Console\Commands;

use Carbon\Carbon;
use App\Models\User;
use App\Models\Lesson;
use Illuminate\Console\Command;
use App\Notifications\LessonNotification;

class CheckLessonDuration extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'lessons:check-duration';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Check and end lessons that have been running for 45 minutes or more';

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $now = Carbon::now();

        $lessons = Lesson::whereIn('status', ['in_progress', 'paused', 'scheduled'])
                        ->with('meeting')
                        ->get();

        foreach ($lessons as $lesson) {
            $meeting = $lesson->meeting;

            if ($lesson->status === 'scheduled') {
                $lessonStart = Carbon::parse($lesson->date . ' ' . $lesson->time);
                
                // Check if the lesson has passed the reminder time (e.g., expired if it's too late)
                if ($now->greaterThanOrEqualTo($lessonStart->addMinutes(15))) {
                    if (!$lesson->teacher_present && !$lesson->student_present) {
                        $lesson->status = 'expired';
                        $lesson->save();

                        $receiver = User::findOrFail($lesson->student->user_id);
                        $receiver->notify(new LessonNotification($lesson, 'expired', 'student', $receiver->id));
                        $receiver = User::findOrFail($lesson->teacher->user_id);
                        $receiver->notify(new LessonNotification($lesson, 'expired', 'teacher', $receiver->id));
                        continue;
                    }
                }

                if (
                    $lessonStart->lt($now) ||
                    $lessonStart->diffInMinutes($now) > 365 || 
                    $lessonStart->diffInMinutes($now) <= 10
                ) {
                    continue; // Skip this lesson
                }
                
                $studentPreference = User::findOrFail($lesson->student->user_id)->notificationPref;
                $studentReminderTime = $studentPreference ? $studentPreference->reminder_time : null;
                $teacherPreference = User::findOrFail($lesson->teacher->user_id)->notificationPref;
                $teacherReminderTime = $teacherPreference ? $teacherPreference->reminder_time : null;

                // Check for the student's reminder time preference
                if ($studentReminderTime && $lessonStart->diffInMinutes($now) === (int) $studentReminderTime) {
                    $receiver = User::findOrFail($lesson->student->user_id);
                    $receiver->notify(new LessonNotification($lesson, 'upcoming', 'student', $receiver->id));
                    $this->info("Sent reminder to student for lesson {$lesson->id}.");
                }

                // Check for the teacher's reminder time preference
                if ($teacherReminderTime && $lessonStart->diffInMinutes($now) === (int) $teacherReminderTime) {
                    $receiver = User::findOrFail($lesson->teacher->user_id);
                    $receiver->notify(new LessonNotification($lesson, 'upcoming', 'teacher', $receiver->id));
                    $this->info("Sent reminder to teacher for lesson {$lesson->id}.");
                }
            }

            // Calculate total duration for lessons in progress or paused
            if ($meeting) {
                $totalDuration = $meeting->total_duration;
                if ($meeting->session_start_time) {
                    $totalDuration += Carbon::now()->diffInMinutes($meeting->session_start_time);
                }

                // Handle lessons in progress exceeding 45 minutes
                if ($lesson->status === 'in_progress' && $totalDuration >= 45) {
                    $lesson->teacher_present = true;
                    $lesson->student_present = true;
                    $lesson->save();
                    $this->info("Ending lesson {$lesson->id} which has been running for {$totalDuration} minutes.");
                    app(\App\Http\Controllers\MeetingController::class)->endMeeting($meeting);

                    $receiver = User::findOrFail($lesson->student->user_id);
                    $receiver->notify(new LessonNotification($lesson, 'completed', 'student', $receiver->id));
                    $receiver = User::findOrFail($lesson->teacher->user_id);
                    $receiver->notify(new LessonNotification($lesson, 'completed', 'teacher', $receiver->id));
                }

                // Handle lessons in paused status for over 1 hour
                if ($lesson->status === 'paused' && $lesson->updated_at->diffInMinutes(Carbon::now()) >= 60) {
                    if ($totalDuration >= 15) {
                        $lesson->status = 'completed';
                        $lesson->teacher_present = true;
                        $lesson->student_present = true;

                        $receiver = User::findOrFail($lesson->student->user_id);
                        $receiver->notify(new LessonNotification($lesson, 'completed', 'student', $receiver->id));
                        $receiver = User::findOrFail($lesson->teacher->user_id);
                        $receiver->notify(new LessonNotification($lesson, 'completed', 'teacher', $receiver->id));
                    } else {
                        $lesson->status = 'expired';

                        $receiver = User::findOrFail($lesson->student->user_id);
                        $receiver->notify(new LessonNotification($lesson, 'expired', 'student', $receiver->id));
                        $receiver = User::findOrFail($lesson->teacher->user_id);
                        $receiver->notify(new LessonNotification($lesson, 'expired', 'teacher', $receiver->id));
                    }
                    $lesson->save();
                }
            }
        }

        $this->info('Lesson duration and notification check completed.');
    }

        
}